<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class SettingsController extends Controller
{
    public function index()
    {
        // Get all settings grouped by category
        $companySettings = Setting::getByGroup('company');
        $emailSettings = Setting::getByGroup('email');
        $paymentSettings = Setting::getByGroup('payment');
        $systemSettings = Setting::getByGroup('system');
        $maintenanceSettings = Setting::getByGroup('maintenance');

        return view('admin.settings.index', compact(
            'companySettings',
            'emailSettings',
            'paymentSettings',
            'systemSettings',
            'maintenanceSettings'
        ));
    }

    public function update(Request $request)
    {
        $group = $request->input('group');

        switch ($group) {
            case 'company':
                return $this->updateCompanySettings($request);
            case 'email':
                return $this->updateEmailSettings($request);
            case 'payment':
                return $this->updatePaymentSettings($request);
            case 'system':
                return $this->updateSystemSettings($request);
            case 'maintenance':
                return $this->updateMaintenanceSettings($request);
            default:
                return redirect()->back()->with('error', 'Invalid settings group.');
        }
    }

    private function updateCompanySettings(Request $request)
    {
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'company_logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
            'company_address' => 'nullable|string|max:500',
            'company_phone' => 'nullable|string|max:20',
            'company_email' => 'nullable|email|max:255',
            'company_website' => 'nullable|url|max:255',
            'company_description' => 'nullable|string|max:1000',
        ]);

        // Handle logo upload
        if ($request->hasFile('company_logo')) {
            $logoPath = $request->file('company_logo')->store('settings', 'uploads');
            Setting::set('company_logo', $logoPath, 'company', 'Company Logo');
        }

        // Update other settings (including empty strings to allow clearing values)
        foreach ($validated as $key => $value) {
            if ($key !== 'company_logo') {
                // Save value even if it's empty string (to allow clearing)
                Setting::set($key, $value ?? '', 'company', ucfirst(str_replace('_', ' ', $key)));
            }
        }

        Setting::clearCache();
        return redirect()->back()->with('success', 'Pengaturan perusahaan berhasil diperbarui!');
    }

    private function updateEmailSettings(Request $request)
    {
        $validated = $request->validate([
            'mail_from_name' => 'nullable|string|max:255',
            'mail_from_address' => 'nullable|email|max:255',
            'mail_host' => 'nullable|string|max:255',
            'mail_port' => 'nullable|integer|min:1|max:65535',
            'mail_username' => 'nullable|string|max:255',
            'mail_password' => 'nullable|string|max:255',
            'mail_encryption' => 'nullable|string|in:tls,ssl',
        ]);

        foreach ($validated as $key => $value) {
            if ($value !== null) {
                Setting::set($key, $value, 'email', ucfirst(str_replace('_', ' ', $key)));
            }
        }

        Setting::clearCache();
        return redirect()->back()->with('success', 'Pengaturan email berhasil diperbarui!');
    }

    private function updatePaymentSettings(Request $request)
    {
        $validated = $request->validate([
            'midtrans_server_key' => 'nullable|string|max:255',
            'midtrans_client_key' => 'nullable|string|max:255',
            'midtrans_is_production' => 'nullable',
            'payment_method_bank_transfer' => 'nullable',
            'payment_method_credit_card' => 'nullable',
            'payment_method_ewallet' => 'nullable',
        ]);

        // Handle checkboxes (they may not be present in request if unchecked)
        $checkboxes = ['midtrans_is_production', 'payment_method_bank_transfer', 'payment_method_credit_card', 'payment_method_ewallet'];
        foreach ($checkboxes as $checkbox) {
            $value = $request->has($checkbox) && $request->input($checkbox) == '1' ? '1' : '0';
            Setting::set($checkbox, $value, 'payment', ucfirst(str_replace('_', ' ', $checkbox)));
        }

        // Handle other fields
        foreach ($validated as $key => $value) {
            if (!in_array($key, $checkboxes) && $value !== null) {
                Setting::set($key, $value, 'payment', ucfirst(str_replace('_', ' ', $key)));
            }
        }

        Setting::clearCache();
        return redirect()->back()->with('success', 'Pengaturan pembayaran berhasil diperbarui!');
    }

    private function updateSystemSettings(Request $request)
    {
        $validated = $request->validate([
            'timezone' => 'nullable|string|max:50',
            'currency' => 'nullable|string|max:10',
            'currency_symbol' => 'nullable|string|max:10',
            'date_format' => 'nullable|string|max:20',
            'time_format' => 'nullable|string|max:20',
            'items_per_page' => 'nullable|integer|min:5|max:100',
        ]);

        foreach ($validated as $key => $value) {
            if ($value !== null) {
                Setting::set($key, $value, 'system', ucfirst(str_replace('_', ' ', $key)));
            }
        }

        Setting::clearCache();
        return redirect()->back()->with('success', 'Pengaturan sistem berhasil diperbarui!');
    }

    private function updateMaintenanceSettings(Request $request)
    {
        $validated = $request->validate([
            'maintenance_mode' => 'nullable',
            'maintenance_message' => 'nullable|string|max:500',
        ]);

        // Handle checkbox (may not be present if unchecked)
        $isEnabled = $request->has('maintenance_mode') && $request->input('maintenance_mode') == '1';
        Setting::setMaintenanceMode($isEnabled);

        if (isset($validated['maintenance_message'])) {
            Setting::set('maintenance_message', $validated['maintenance_message'], 'maintenance', 'Maintenance Message');
        }

        Setting::clearCache();

        // Enable/Disable Laravel maintenance mode
        try {
            if ($isEnabled) {
                // Generate a secret that admin can use to bypass maintenance
                try {
                    $secret = bin2hex(random_bytes(16));
                } catch (\Exception $e) {
                    $secret = substr(md5(uniqid('', true)), 0, 32);
                }

                // Save secret to settings so it can be shown in the UI
                Setting::set('maintenance_secret', $secret, 'maintenance', 'Maintenance Secret');

                // Optionally allow current admin IP for convenience
                $allowIp = $request->ip();

                Artisan::call('down', [
                    '--secret' => $secret,
                    '--allow' => $allowIp,
                    '--render' => 'errors::503'
                ]);
            } else {
                // Clear any stored secret when turning off maintenance
                Setting::set('maintenance_secret', '', 'maintenance', 'Maintenance Secret');
                Artisan::call('up');
            }
        } catch (\Exception $e) {
            // If maintenance mode command fails, continue anyway
            Log::warning('Failed to toggle Laravel maintenance mode: ' . $e->getMessage());
        }

        return redirect()->back()->with('success', 'Mode pemeliharaan berhasil ' . ($isEnabled ? 'diaktifkan' : 'dinonaktifkan') . '!');
    }
}
