<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Sale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CustomerConfirmationController extends Controller
{
    /**
     * Show confirmation page for customer
     */
    public function show($id)
    {
        $order = Order::findOrFail($id);
        $order->load(['product', 'production', 'handoverBy']);
        
        // Check if already confirmed FIRST (before checking delivery status)
        // Because after confirmation, handover_status changes to 'confirmed', not 'delivered'
        if ($order->customer_confirmed_at) {
            return view('user.orders.already-confirmed', compact('order'));
        }

        // Check if order is delivered (ready for confirmation)
        if ($order->handover_status !== 'delivered') {
            // Show informative page instead of redirecting
            return view('user.orders.not-ready-for-confirmation', compact('order'));
        }

        // Order is delivered and not yet confirmed - show confirmation form
        return view('user.orders.confirm-delivery', compact('order'));
    }

    /**
     * Customer confirms delivery
     */
    public function confirm(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        
        // Check if order can be confirmed
        if (!$order->canCustomerConfirm()) {
            return back()->withErrors(['status' => 'Pesanan tidak dapat dikonfirmasi.']);
        }

        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'feedback' => 'nullable|string|max:2000',
        ]);

        return DB::transaction(function () use ($request, $order) {
            // Update order status
            $order->update([
                'handover_status' => 'confirmed',
                'customer_confirmed_at' => now(),
                'customer_rating' => $request->rating,
                'customer_feedback' => $request->feedback,
                'status' => 'selesai', // Final payment status
            ]);

            // Reload order untuk mendapatkan data terbaru
            $order->refresh();
            
            // Validasi: Sale hanya dibuat jika status pembayaran = selesai DAN status produksi = selesai
            // Ini memastikan bahwa penjualan hanya terjadi ketika semua proses selesai
            $canCreateSale = $order->status === 'selesai' && 
                            $order->production_status === 'selesai' &&
                            !$order->sale;

            // Create sale record jika semua kondisi terpenuhi
            if ($canCreateSale) {
                Sale::create([
                    'order_id' => $order->id,
                    'product_id' => $order->product_id,
                    'user_id' => $order->user_id,
                    'quantity' => $order->quantity,
                    'unit_price' => $order->price,
                    'total_price' => $order->total_price,
                    'sale_date' => now(),
                    'payment_method' => $order->payment_method,
                    'status' => 'selesai',
                    'notes' => 'Penjualan selesai - Dikonfirmasi customer setelah serah terima',
                ]);
                
                // Kirim notifikasi ke admin
                if ($order->production) {
                    try {
                        $notificationService = new \App\Services\NotificationService();
                        $notificationService->notifyProductionCompletedAndSold($order->production->fresh());
                    } catch (\Exception $e) {
                        \Log::error('Failed to send production completed and sold notification: ' . $e->getMessage());
                    }
                }
                
                \Log::info('Sale created after customer handover confirmation', [
                    'order_id' => $order->id,
                    'production_id' => $order->production?->id,
                    'production_status' => $order->production_status,
                    'payment_status' => $order->status,
                    'handover_status' => $order->handover_status,
                    'customer_name' => $order->customer_name,
                    'customer_email' => $order->customer_email,
                    'sale_date' => now()->toDateTimeString(),
                    'total_price' => $order->total_price,
                ]);
            } else {
                // Log jika sale tidak bisa dibuat
                \Log::warning('Sale not created after customer confirmation - conditions not met', [
                    'order_id' => $order->id,
                    'production_id' => $order->production?->id,
                    'payment_status' => $order->status,
                    'production_status' => $order->production_status,
                    'handover_status' => $order->handover_status,
                    'sale_exists' => $order->sale ? true : false,
                    'customer_confirmed_at' => $order->customer_confirmed_at?->toDateTimeString(),
                ]);
            }

            return redirect()->route('user.orders.confirmation-success', $order)
                ->with('success', 'Terima kasih! Pesanan Anda telah dikonfirmasi.');
        });
    }

    /**
     * Show confirmation success page
     */
    public function success($id)
    {
        $order = Order::findOrFail($id);
        $order->load(['product', 'sale']);

        return view('user.orders.confirmation-success', compact('order'));
    }
}

