<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Sale;
use App\Models\Purchase;
use App\Models\Product;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\User;
use App\Models\Order;
use App\Models\Receivable;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();
        
        switch ($user->user_type) {
            case 'admin':
                return $this->adminDashboard();
            case 'supervisor':
                return $this->supervisorDashboard();
            case 'teknisi':
                return $this->teknisiDashboard();
            case 'user':
            default:
                return view('dashboard');
        }
    }

    private function adminDashboard()
    {
        $stats = $this->getAdminStats();
        $chartData = $this->getChartData();
        $recentData = $this->getRecentData();
        
        return view('admin.dashboard', compact('stats', 'chartData', 'recentData'));
    }

    public function supervisorDashboard()
    {
        $stats = $this->getSupervisorStats();
        
        return view('supervisor.dashboard', compact('stats'));
    }

    public function teknisiDashboard()
    {
        $stats = $this->getTeknisiStats();
        $chartData = $this->getTeknisiChartData();
        
        return view('teknisi.dashboard', compact('stats', 'chartData'));
    }

    private function getAdminStats()
    {
        $now = Carbon::now();
        $lastMonth = $now->copy()->subMonth();
        
        // Total Penjualan Bulan Ini
        $currentMonthSales = Sale::whereMonth('sale_date', $now->month)
            ->whereYear('sale_date', $now->year)
            ->where('status', 'selesai')
            ->sum('total_price');
            
        $lastMonthSales = Sale::whereMonth('sale_date', $lastMonth->month)
            ->whereYear('sale_date', $lastMonth->year)
            ->where('status', 'selesai')
            ->sum('total_price');
            
        $salesGrowth = $lastMonthSales > 0 ? (($currentMonthSales - $lastMonthSales) / $lastMonthSales) * 100 : 0;

        // Total Pembelian Bulan Ini
        $currentMonthPurchases = Purchase::whereMonth('purchase_date', $now->month)
            ->whereYear('purchase_date', $now->year)
            ->sum('total_amount');
            
        $lastMonthPurchases = Purchase::whereMonth('purchase_date', $lastMonth->month)
            ->whereYear('purchase_date', $lastMonth->year)
            ->sum('total_amount');
            
        $purchasesGrowth = $lastMonthPurchases > 0 ? (($currentMonthPurchases - $lastMonthPurchases) / $lastMonthPurchases) * 100 : 0;

        // Total Produk Aktif
        $totalProducts = Product::where('is_active', true)->count();
        $lastMonthProducts = Product::where('is_active', true)
            ->where('created_at', '<=', $lastMonth->endOfMonth())
            ->count();
            
        $productsGrowth = $lastMonthProducts > 0 ? (($totalProducts - $lastMonthProducts) / $lastMonthProducts) * 100 : 0;

        // Total Order Bulan Ini
        $currentMonthOrders = Order::whereMonth('created_at', $now->month)
            ->whereYear('created_at', $now->year)
            ->count();
            
        $lastMonthOrders = Order::whereMonth('created_at', $lastMonth->month)
            ->whereYear('created_at', $lastMonth->year)
            ->count();
            
        $ordersGrowth = $lastMonthOrders > 0 ? (($currentMonthOrders - $lastMonthOrders) / $lastMonthOrders) * 100 : 0;

        // Production Statistics dengan Planning Status
        $totalProductions = \App\Models\Production::count();
        $pendingPlanningProductions = \App\Models\Production::where('planning_status', 'planning')->count();
        $pendingApprovalProductions = \App\Models\Production::where('planning_status', 'pending_approval')->count();
        $approvedProductions = \App\Models\Production::where('planning_status', 'approved')->count();
        $activeProductions = \App\Models\Production::where('status', 'dalam_proses')->count();
        $completedProductions = \App\Models\Production::where('status', 'selesai')->count();

        // Stock Alert (Materials & Spareparts dengan stock rendah)
        $lowStockMaterials = Material::where('is_active', true)
            ->where('stock', '<=', 10)
            ->count();
            
        $lowStockSpareparts = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5)
            ->count();
            
        $totalLowStock = $lowStockMaterials + $lowStockSpareparts;

        // Total Inventory Value (Materials + Spareparts)
        $totalMaterialsValue = Material::where('is_active', true)
            ->get()
            ->sum(function($material) {
                return $material->stock * $material->price;
            });
            
        $totalSparepartsValue = Sparepart::where('is_active', true)
            ->get()
            ->sum(function($sparepart) {
                return $sparepart->stock * $sparepart->price;
            });
            
        $totalInventoryValue = $totalMaterialsValue + $totalSparepartsValue;

        // Data Piutang
        $totalReceivables = Receivable::where('status', 'active')->count();
        $overdueReceivables = Receivable::overdue()->count();
        $totalReceivableAmount = Receivable::where('status', 'active')->sum('remaining_amount');
        $paidReceivableAmount = Receivable::where('status', 'active')->sum('paid_amount');

        // Order yang sudah diserahkan (delivered) tapi belum dikonfirmasi customer
        // Hanya order dengan handover_status = 'delivered' yang menunggu konfirmasi customer
        $completedProductionPendingConfirmation = Order::whereHas('production', function($q) {
                $q->where('status', 'selesai')
                  ->where('completion_status', 'approved');
            })
            ->where('handover_status', 'delivered')
            ->whereNull('customer_confirmed_at')
            ->count();

        // Order Status Summary (menggunakan order_status)
        // order_status hanya memiliki: menunggu, diterima, ditolak
        // Untuk order yang selesai secara keseluruhan, gunakan handover_status = 'confirmed'
        $orderStatusSummary = [
            'menunggu' => Order::where('order_status', 'menunggu')->count(),
            'diterima' => Order::where('order_status', 'diterima')->count(),
            'ditolak' => Order::where('order_status', 'ditolak')->count(),
            'selesai' => Order::where('handover_status', 'confirmed')->count(), // Order yang sudah dikonfirmasi customer
        ];

        // Payment Status Summary
        $paymentStatusSummary = [
            'menunggu_verifikasi' => Order::where('status', 'menunggu_verifikasi')->count(),
            'diproses' => Order::where('status', 'diproses')->count(),
            'selesai' => Order::where('status', 'selesai')->count(),
            'gagal' => Order::where('status', 'gagal')->count(),
        ];

        // Handover Status Summary
        $handoverStatusSummary = [
            'pending' => Order::where('handover_status', 'pending')->orWhereNull('handover_status')->count(),
            'ready' => Order::where('handover_status', 'ready')->count(),
            'delivered' => Order::where('handover_status', 'delivered')->count(),
            'confirmed' => Order::where('handover_status', 'confirmed')->count(),
        ];

        // Production Status Summary
        $productionStatusSummary = [
            'menunggu' => Order::where('production_status', 'menunggu')->count(),
            'dalam_proses' => Order::where('production_status', 'dalam_proses')->count(),
            'selesai' => Order::where('production_status', 'selesai')->count(),
            'dibatalkan' => Order::where('production_status', 'dibatalkan')->count(),
        ];

        // Customer Statistics
        $totalCustomers = User::where('user_type', 'user')->count();
        $newCustomersThisMonth = User::where('user_type', 'user')
            ->whereMonth('created_at', $now->month)
            ->whereYear('created_at', $now->year)
            ->count();

        // Order by Payment Method
        $orderByPaymentMethod = Order::select('payment_method', DB::raw('COUNT(*) as count'))
            ->groupBy('payment_method')
            ->get()
            ->pluck('count', 'payment_method')
            ->toArray();

        // Profit Calculation (Sales - Purchases - Production Costs)
        $totalProductionCosts = \App\Models\Production::where('status', 'selesai')
            ->sum(DB::raw('COALESCE(total_production_cost, 0) + COALESCE(labor_cost, 0)'));
        
        $grossProfit = $currentMonthSales - $currentMonthPurchases;
        $netProfit = $grossProfit - $totalProductionCosts;

        // Pending Tasks
        $pendingTasks = [
            'unread_orders' => Order::where('is_read', false)->count(),
            'pending_handover' => $completedProductionPendingConfirmation,
            'pending_production_approvals' => \App\Models\Production::where('planning_status', 'pending_approval')->count(),
            'pending_completion_approvals' => \App\Models\Production::where('completion_status', 'pending_approval')->count(),
        ];
        $totalPendingTasks = array_sum($pendingTasks);

        return [
            'total_sales' => $currentMonthSales,
            'sales_growth' => $salesGrowth,
            'total_purchases' => $currentMonthPurchases,
            'purchases_growth' => $purchasesGrowth,
            'total_products' => $totalProducts,
            'products_growth' => $productsGrowth,
            'total_orders' => $currentMonthOrders,
            'orders_growth' => $ordersGrowth,
            'low_stock_materials' => $lowStockMaterials,
            'low_stock_spareparts' => $lowStockSpareparts,
            'total_low_stock' => $totalLowStock,
            'total_inventory_value' => $totalInventoryValue,
            'total_receivables' => $totalReceivables,
            'overdue_receivables' => $overdueReceivables,
            'total_receivable_amount' => $totalReceivableAmount,
            'paid_receivable_amount' => $paidReceivableAmount,
            'receivable_percentage' => $totalReceivableAmount > 0 ? ($paidReceivableAmount / $totalReceivableAmount) * 100 : 0,
            'completed_production_pending_confirmation' => $completedProductionPendingConfirmation,
            // Production Statistics
            'total_productions' => $totalProductions,
            'pending_planning_productions' => $pendingPlanningProductions,
            'pending_approval_productions' => $pendingApprovalProductions,
            'approved_productions' => $approvedProductions,
            'active_productions' => $activeProductions,
            'completed_productions' => $completedProductions,
            // Status Summaries
            'order_status_summary' => $orderStatusSummary,
            'payment_status_summary' => $paymentStatusSummary,
            'handover_status_summary' => $handoverStatusSummary,
            'production_status_summary' => $productionStatusSummary,
            // Customer Statistics
            'total_customers' => $totalCustomers,
            'new_customers_this_month' => $newCustomersThisMonth,
            // Payment Method Distribution
            'order_by_payment_method' => $orderByPaymentMethod,
            // Profit Summary
            'gross_profit' => $grossProfit,
            'net_profit' => $netProfit,
            'total_production_costs' => $totalProductionCosts,
            // Pending Tasks
            'pending_tasks' => $pendingTasks,
            'total_pending_tasks' => $totalPendingTasks,
        ];
    }

    private function getChartData()
    {
        // Data penjualan 12 bulan terakhir
        $salesData = Sale::select(
                DB::raw('MONTH(sale_date) as month'),
                DB::raw('YEAR(sale_date) as year'),
                DB::raw('SUM(total_price) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('sale_date', '>=', Carbon::now()->subMonths(12))
            ->where('status', 'selesai')
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Data pembelian 12 bulan terakhir
        $purchasesData = Purchase::select(
                DB::raw('MONTH(purchase_date) as month'),
                DB::raw('YEAR(purchase_date) as year'),
                DB::raw('SUM(total_amount) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('purchase_date', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Data users 12 bulan terakhir
        $usersData = User::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('COUNT(*) as count')
            )
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Data kategori produk
        $categoriesData = Product::with('category')
            ->where('is_active', true)
            ->get()
            ->groupBy('category.name')
            ->map(function($products) {
                return $products->count();
            });

        // Map data ke array 12 bulan terakhir dengan benar
        $now = Carbon::now();
        $salesArray = [];
        $purchasesArray = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $targetDate = $now->copy()->subMonths($i);
            $targetMonth = $targetDate->month;
            $targetYear = $targetDate->year;
            
            $salesValue = $salesData->first(function($item) use ($targetMonth, $targetYear) {
                return $item->month == $targetMonth && $item->year == $targetYear;
            });
            $salesArray[] = $salesValue ? (float)$salesValue->total : 0;
            
            $purchasesValue = $purchasesData->first(function($item) use ($targetMonth, $targetYear) {
                return $item->month == $targetMonth && $item->year == $targetYear;
            });
            $purchasesArray[] = $purchasesValue ? (float)$purchasesValue->total : 0;
        }

        return [
            'sales' => collect($salesArray),
            'purchases' => collect($purchasesArray),
            'users' => $usersData,
            'categories' => $categoriesData,
        ];
    }

    private function getRecentData()
    {
        // Recent Orders
        $recentOrders = Order::with(['product', 'user'])
            ->latest()
            ->limit(5)
            ->get();
        
        // Recent Inquiries (unread priority)
        $recentInquiries = \App\Models\ProductInquiry::with('product')
            ->orderBy('is_read', 'asc')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
        
        // Active Productions
        $activeProductions = \App\Models\Production::with(['order', 'product', 'teknisi'])
            ->where('status', 'dalam_proses')
            ->latest()
            ->limit(5)
            ->get();
        
        // Pending Approvals
        $pendingApprovals = \App\Models\Production::where('planning_status', 'pending_approval')
            ->orWhere('completion_status', 'pending_approval')
            ->count();
        
        // Top Products (by order count)
        $topProducts = Order::select('product_id', DB::raw('COUNT(*) as order_count'))
            ->with('product:id,product_title')
            ->groupBy('product_id')
            ->orderBy('order_count', 'desc')
            ->limit(5)
            ->get();
        
        return [
            'recent_orders' => $recentOrders,
            'recent_inquiries' => $recentInquiries,
            'active_productions' => $activeProductions,
            'pending_approvals' => $pendingApprovals,
            'top_products' => $topProducts,
        ];
    }

    private function getSupervisorStats()
    {
        $now = Carbon::now();
        
        // Pending Approvals
        $pendingOrders = Order::where('order_status', 'menunggu')->count();
        $pendingProductionApprovals = \App\Models\Production::where('planning_status', 'pending_approval')->count();
        $pendingPurchaseApprovals = Purchase::where('status', 'pending')->count();
        
        // Total pending approvals
        $totalPendingApprovals = $pendingOrders + $pendingProductionApprovals + $pendingPurchaseApprovals;
        
        // Recent activities
        $recentOrders = Order::with(['product', 'user'])
            ->latest()
            ->limit(5)
            ->get();
            
        $recentProductions = \App\Models\Production::with(['order', 'product'])
            ->latest()
            ->limit(5)
            ->get();
        
        // Monthly overview
        $monthlySales = Sale::whereMonth('sale_date', $now->month)
            ->whereYear('sale_date', $now->year)
            ->where('status', 'selesai')
            ->sum('total_price');
            
        $monthlyOrders = Order::whereMonth('created_at', $now->month)
            ->whereYear('created_at', $now->year)
            ->count();

        return [
            'pending_orders' => $pendingOrders,
            'pending_production_approvals' => $pendingProductionApprovals,
            'pending_purchase_approvals' => $pendingPurchaseApprovals,
            'total_pending_approvals' => $totalPendingApprovals,
            'recent_orders' => $recentOrders,
            'recent_productions' => $recentProductions,
            'monthly_sales' => $monthlySales,
            'monthly_orders' => $monthlyOrders,
        ];
    }

    private function getTeknisiStats()
    {
        $now = Carbon::now();
        $teknisiId = Auth::id();
        
        // My Productions dengan Planning Status
        $myPlanningProductions = \App\Models\Production::where('teknisi_id', $teknisiId)
            ->where('planning_status', 'planning')->count();
        $myPendingApprovals = \App\Models\Production::where('teknisi_id', $teknisiId)
            ->where('planning_status', 'pending_approval')->count();
        $myActiveProductions = \App\Models\Production::where('teknisi_id', $teknisiId)
            ->where('status', 'dalam_proses')->count();
        $completedProductions = \App\Models\Production::where('teknisi_id', $teknisiId)
            ->where('status', 'selesai')
            ->whereMonth('updated_at', $now->month)
            ->whereYear('updated_at', $now->year)
            ->count();
            
        // Materials & Spareparts
        $lowStockMaterials = Material::where('is_active', true)
            ->where('stock', '<=', 10)
            ->count();
            
        $lowStockSpareparts = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5)
            ->count();
        
        // Recent productions (hanya yang ditugaskan ke teknisi ini)
        $recentProductions = \App\Models\Production::with(['order', 'product'])
            ->where('teknisi_id', $teknisiId)
            ->where('status', '!=', 'selesai')
            ->latest()
            ->limit(5)
            ->get();
            
        // Recent history (produksi selesai)
        $recentHistory = \App\Models\Production::with(['order', 'product'])
            ->where('teknisi_id', $teknisiId)
            ->where('status', 'selesai')
            ->latest()
            ->limit(5)
            ->get();
            
        // Materials usage this month
        $monthlyMaterialUsage = \App\Models\ProductionMaterial::whereHas('production', function($q) use ($teknisiId) {
                $q->where('teknisi_id', $teknisiId);
            })
            ->whereMonth('created_at', $now->month)
            ->whereYear('created_at', $now->year)
            ->sum('total_cost');
            
        $monthlySparepartUsage = \App\Models\ProductionSparepart::whereHas('production', function($q) use ($teknisiId) {
                $q->where('teknisi_id', $teknisiId);
            })
            ->whereMonth('created_at', $now->month)
            ->whereYear('created_at', $now->year)
            ->sum('total_cost');

        // Production Status Summary (hanya untuk teknisi ini)
        $productionStatusSummary = [
            'planning' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('planning_status', 'planning')->count(),
            'pending_approval' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('planning_status', 'pending_approval')->count(),
            'approved' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('planning_status', 'approved')->count(),
            'rejected' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('planning_status', 'rejected')->count(),
        ];

        // Production Completion Status Summary
        $completionStatusSummary = [
            'pending' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('completion_status', 'pending')->orWhereNull('completion_status')->count(),
            'pending_approval' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('completion_status', 'pending_approval')->count(),
            'approved' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('completion_status', 'approved')->count(),
            'rejected' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('completion_status', 'rejected')->count(),
        ];

        // Production Status (menunggu, dalam_proses, selesai)
        $productionStatusCount = [
            'menunggu' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('status', 'menunggu')->count(),
            'dalam_proses' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('status', 'dalam_proses')->count(),
            'selesai' => \App\Models\Production::where('teknisi_id', $teknisiId)->where('status', 'selesai')->count(),
        ];

        // Pending Quotations (inquiries yang perlu estimasi teknisi)
        // Status 'teknisi_review' berarti menunggu estimasi dari teknisi
        $pendingQuotations = \App\Models\ProductInquiry::where('status', 'teknisi_review')
            ->whereNull('teknisi_estimated_at')
            ->count();

        return [
            'my_planning_productions' => $myPlanningProductions,
            'my_pending_approvals' => $myPendingApprovals,
            'my_active_productions' => $myActiveProductions,
            'completed_productions' => $completedProductions,
            'low_stock_materials' => $lowStockMaterials,
            'low_stock_spareparts' => $lowStockSpareparts,
            'recent_productions' => $recentProductions,
            'recent_history' => $recentHistory,
            'monthly_material_usage' => $monthlyMaterialUsage,
            'monthly_sparepart_usage' => $monthlySparepartUsage,
            'production_status_summary' => $productionStatusSummary,
            'completion_status_summary' => $completionStatusSummary,
            'production_status_count' => $productionStatusCount,
            'pending_quotations' => $pendingQuotations,
            'total_low_stock' => $lowStockMaterials + $lowStockSpareparts,
        ];
    }

    private function getTeknisiChartData()
    {
        $teknisiId = Auth::id();
        $now = Carbon::now();
        
        // Data produksi selesai per bulan (12 bulan terakhir)
        $productionsData = \App\Models\Production::select(
                DB::raw('MONTH(updated_at) as month'),
                DB::raw('YEAR(updated_at) as year'),
                DB::raw('COUNT(*) as count')
            )
            ->where('teknisi_id', $teknisiId)
            ->where('status', 'selesai')
            ->where('updated_at', '>=', $now->copy()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Data penggunaan bahan per bulan
        $materialsUsageData = \App\Models\ProductionMaterial::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('SUM(total_cost) as total')
            )
            ->whereHas('production', function($q) use ($teknisiId) {
                $q->where('teknisi_id', $teknisiId);
            })
            ->where('created_at', '>=', $now->copy()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Data penggunaan sparepart per bulan
        $sparepartsUsageData = \App\Models\ProductionSparepart::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('SUM(total_cost) as total')
            )
            ->whereHas('production', function($q) use ($teknisiId) {
                $q->where('teknisi_id', $teknisiId);
            })
            ->where('created_at', '>=', $now->copy()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Status produksi saat ini (untuk pie chart)
        $statusData = \App\Models\Production::where('teknisi_id', $teknisiId)
            ->where('status', '!=', 'selesai')
            ->select('status', DB::raw('COUNT(*) as count'))
            ->groupBy('status')
            ->get()
            ->pluck('count', 'status');

        // Map data ke array 12 bulan terakhir
        $productionsArray = [];
        $materialsArray = [];
        $sparepartsArray = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $targetDate = $now->copy()->subMonths($i);
            $targetMonth = $targetDate->month;
            $targetYear = $targetDate->year;
            
            $productionValue = $productionsData->first(function($item) use ($targetMonth, $targetYear) {
                return $item->month == $targetMonth && $item->year == $targetYear;
            });
            $productionsArray[] = $productionValue ? (int)$productionValue->count : 0;
            
            $materialValue = $materialsUsageData->first(function($item) use ($targetMonth, $targetYear) {
                return $item->month == $targetMonth && $item->year == $targetYear;
            });
            $materialsArray[] = $materialValue ? (float)$materialValue->total : 0;
            
            $sparepartValue = $sparepartsUsageData->first(function($item) use ($targetMonth, $targetYear) {
                return $item->month == $targetMonth && $item->year == $targetYear;
            });
            $sparepartsArray[] = $sparepartValue ? (float)$sparepartValue->total : 0;
        }

        return [
            'productions' => collect($productionsArray),
            'materials_usage' => collect($materialsArray),
            'spareparts_usage' => collect($sparepartsArray),
            'status_distribution' => $statusData,
        ];
    }
}
