<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Category;
use App\Models\Product;
use App\Models\Hero;
use Illuminate\Support\Facades\Mail;
use App\Models\Contact;
use App\Mail\ContactMail;

class HomeController extends Controller
{
    public function index()
    {
        if (Auth::user()->user_type === "admin") {
            return view('admin.dashboard');
        }

        return view('dashboard');
    }

    // home
    public function home()
    {
        // Kategori besar (parent) untuk navbar - hanya yang type service
        $productCategories = Category::where('type', 'service')
            ->whereNull('parent_id')
            ->with('children')
            ->orderBy('name')
            ->take(8)
            ->get();

        // Semua sub kategori (children) untuk section kategori di homepage
        $allCategories = Category::where('type', 'service')
            ->whereNotNull('parent_id')
            ->with('parent')
            ->orderBy('name')
            ->get();

        // Ambil 10 produk terbaru (HANYA yang aktif)
        $products = Product::where('is_active', true)
            ->with(['category', 'images'])
            ->latest('id')
            ->take(10)
            ->get();

        // Ambil heroes yang aktif, diurutkan berdasarkan order
        $heroes = Hero::where('is_active', true)
            ->orderBy('order')
            ->orderBy('created_at', 'desc')
            ->get();

        // Statistik untuk Pencapaian Kami
        // 1. Total Mesin Diproduksi (berdasarkan jumlah produk aktif)
        // Minimum 40 untuk branding
        $actualProducts = \App\Models\Product::where('is_active', true)->count();
        $totalProducts = max(40, $actualProducts); // Minimum 40, atau lebih jika ada lebih banyak
        
        // 2. Tahun Pengalaman (dinamis berdasarkan tahun mulai, default 2017)
        $startYear = 2017; // Tahun mulai bisnis
        $currentYear = date('Y');
        $yearsOfExperience = $currentYear - $startYear;
        
        // 3. Kerja Sama (fixed value 10 untuk branding)
        // Nilai tetap 10 untuk branding, tidak berdasarkan hitungan aktual
        $totalCooperations = 10;
        
        // Produk Terlaris berdasarkan jumlah order yang diterima
        $bestSellingProducts = \App\Models\Product::where('is_active', true)
            ->with(['category', 'images'])
            ->withCount(['orders as sold_count' => function($query) {
                $query->where('order_status', 'diterima');
            }])
            ->orderBy('sold_count', 'desc')
            ->orderBy('created_at', 'desc')
            ->take(10)
            ->get()
            ->filter(function($product) {
                return ($product->sold_count ?? 0) > 0;
            })
            ->map(function($product) {
                $product->sold = $product->sold_count ?? 0;
                return $product;
            })
            ->values(); // Re-index array setelah filter

        return view('home.index', compact(
            'productCategories', 
            'allCategories', 
            'products', 
            'heroes',
            'totalProducts',
            'yearsOfExperience',
            'totalCooperations',
            'bestSellingProducts'
        ));
    }

    // kategori ambil products
    public function kategori($id)
    {
        $category = Category::findOrFail($id);

        // Query produk berdasarkan kategori
        $productQuery = Product::where('is_active', true);

        // Jika parent category, ambil produk dari parent + semua sub kategori
        if ($category->isParent()) {
            $childCategoryIds = $category->children()->pluck('id')->toArray();
            $allCategoryIds = array_merge([$id], $childCategoryIds);
            $productQuery->whereIn('category_id', $allCategoryIds);
        } else {
            // Jika sub category, ambil produk dari sub category saja
            $productQuery->where('category_id', $id);
        }

        $products = $productQuery->with(['category', 'images'])
            ->latest('id')
            ->paginate(10);

        return view('home.kategori', compact('category', 'products'));
    }

    // Tampilkan halaman kontak
    public function kontak()
    {
        return view('home.kontak');
    }

    // Simpan data kontak dan kirim email ke Mailtrap
    public function kontakStore(Request $request)
    {
        $request->validate([
            'nama'  => 'required|string|max:100',
            'email' => 'required|email',
            'pesan' => 'required|string',
        ]);

        // Simpan ke database
        $contact = Contact::create($request->all());

        // Kirim notifikasi untuk kontak baru
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyNewContact($contact, true);
        } catch (\Exception $e) {
            \Log::error('Failed to send contact notification: ' . $e->getMessage());
        }

        // Kirim email ke Mailtrap sandbox
        try {
            Mail::to('6479d7f8d7-c134ee+user1@inbox.mailtrap.io')
                ->send(new ContactMail($contact));
        } catch (\Exception $e) {
            dd('Email gagal dikirim: ' . $e->getMessage());
        }

        return back()->with('success', 'Pesan berhasil dikirim!');
    }

    // konsultasi
    public function konsultasi()
    {
        return view('home.konsultasi');
    }

    // kebijakan
    public function kebijakan()
    {
        return view('home.kebijakan');
    }

    // all produk grid
    public function all(Request $request)
    {
        // Tentukan type yang dipilih (services, atau all)
        $selectedType = $request->get('type', 'services');

        // Ambil sub kategori untuk filter, dikelompokkan berdasarkan parent
        $subCategories = Category::where('type', 'service')
            ->whereNotNull('parent_id') // Hanya sub kategori
            ->with('parent')
            ->orderBy('name', 'asc')
            ->get();
        
        // Kelompokkan berdasarkan parent untuk tampilan yang lebih rapi
        $categoriesGrouped = $subCategories->groupBy('parent_id');
        
        // Ambil semua parent untuk referensi dengan jumlah produk
        $parentCategories = Category::where('type', 'service')
            ->whereNull('parent_id')
            ->orderBy('name')
            ->get()
            ->map(function ($parent) {
                // Hitung total produk di parent + semua sub kategori
                $parentProductCount = Product::where('is_active', true)
                    ->where('category_id', $parent->id)
                    ->count();
                $subCategoryIds = $parent->children()->pluck('id')->toArray();
                $subProductCount = Product::where('is_active', true)
                    ->whereIn('category_id', $subCategoryIds)
                    ->count();
                $parent->total_product_count = $parentProductCount + $subProductCount;
                
                // Hitung jumlah produk per sub kategori
                $parent->children = $parent->children()->get()->map(function ($child) {
                    $child->product_count = Product::where('is_active', true)
                        ->where('category_id', $child->id)
                        ->count();
                    return $child;
                });
                
                return $parent;
            });

        // QUERY PRODUK (HANYA yang aktif)
        $productQuery = Product::where('is_active', true);

        // Filter kategori produk berdasarkan category_id
        if ($request->has('category') && $request->category != '') {
            $categoryId = $request->category;
            
            // Cek apakah category_id adalah parent atau sub category
            $selectedCategory = Category::find($categoryId);
            
            if ($selectedCategory) {
                if ($selectedCategory->isParent()) {
                    // Jika parent category, ambil produk dari parent + semua sub kategori
                    $childCategoryIds = $selectedCategory->children()->pluck('id')->toArray();
                    $allCategoryIds = array_merge([$categoryId], $childCategoryIds);
                    $productQuery->whereIn('category_id', $allCategoryIds);
                } else {
                    // Jika sub category, ambil produk dari sub category saja
                    $productQuery->where('category_id', $categoryId);
                }
            }
        }

        // Filter pencarian produk
        if ($request->has('search') && $request->search != '') {
            $searchTerm = $request->search;
            $productQuery->where(function ($q) use ($searchTerm) {
                $q->where('product_title', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('product_description', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhereHas('category', function ($query) use ($searchTerm) {
                        $query->where('name', 'LIKE', '%' . $searchTerm . '%');
                    });
            });
        }

        // Filter harga produk
        if ($request->has('min_price') && $request->min_price != '') {
            $productQuery->where('product_prices', '>=', $request->min_price);
        }

        if ($request->has('max_price') && $request->max_price != '') {
            $productQuery->where('product_prices', '<=', $request->max_price);
        }

        // Sorting produk
        $sortBy = $request->get('sort', 'newest');
        switch ($sortBy) {
            case 'price_low':
                $productQuery->orderBy('product_prices', 'asc');
                break;
            case 'price_high':
                $productQuery->orderBy('product_prices', 'desc');
                break;
            case 'name':
                $productQuery->orderBy('product_title', 'asc');
                break;
            case 'oldest':
                $productQuery->orderBy('created_at', 'asc');
                break;
            default: // newest
                $productQuery->orderBy('created_at', 'desc');
                break;
        }

        // Hitung total items sebelum pagination
        $totalItems = $productQuery->count();

        // Load relasi yang diperlukan dan paginate
        $products = $productQuery->with([
            'images',
            'category',
            'specifications'
        ])->paginate(12)->onEachSide(3)->withQueryString();

        // Data untuk view
        $selectedCategory = $request->get('category', '');

        return view('home.all', compact(
            'products',
            'subCategories',
            'categoriesGrouped',
            'parentCategories',
            'totalItems',
            'selectedCategory',
            'selectedType'
        ));
    }

    // Halaman Produk Terlaris (Terpisah dari all)
    public function bestSelling(Request $request)
    {
        // Ambil sub kategori untuk filter, dikelompokkan berdasarkan parent
        $subCategories = Category::where('type', 'service')
            ->whereNotNull('parent_id')
            ->with('parent')
            ->orderBy('name', 'asc')
            ->get();
        
        // Ambil semua parent untuk referensi dengan jumlah produk
        $parentCategories = Category::where('type', 'service')
            ->whereNull('parent_id')
            ->orderBy('name')
            ->get()
            ->map(function ($parent) {
                $parentProductCount = Product::where('is_active', true)
                    ->where('category_id', $parent->id)
                    ->count();
                $subCategoryIds = $parent->children()->pluck('id')->toArray();
                $subProductCount = Product::where('is_active', true)
                    ->whereIn('category_id', $subCategoryIds)
                    ->count();
                $parent->total_product_count = $parentProductCount + $subProductCount;
                
                $parent->children = $parent->children()->get()->map(function ($child) {
                    $child->product_count = Product::where('is_active', true)
                        ->where('category_id', $child->id)
                        ->count();
                    return $child;
                });
                
                return $parent;
            });

        // QUERY PRODUK TERLARIS (HANYA yang aktif dan memiliki order)
        $productQuery = Product::where('is_active', true)
            ->withCount(['orders as sold_count' => function($query) {
                $query->where('order_status', 'diterima');
            }])
            ->having('sold_count', '>', 0) // Hanya produk yang sudah terjual
            ->orderBy('sold_count', 'desc')
            ->orderBy('created_at', 'desc');

        // Filter kategori produk
        if ($request->has('category') && $request->category != '') {
            $categoryId = $request->category;
            $selectedCategory = Category::find($categoryId);
            
            if ($selectedCategory) {
                if ($selectedCategory->isParent()) {
                    $childCategoryIds = $selectedCategory->children()->pluck('id')->toArray();
                    $allCategoryIds = array_merge([$categoryId], $childCategoryIds);
                    $productQuery->whereIn('category_id', $allCategoryIds);
                } else {
                    $productQuery->where('category_id', $categoryId);
                }
            }
        }

        // Filter pencarian produk
        if ($request->has('search') && $request->search != '') {
            $searchTerm = $request->search;
            $productQuery->where(function ($q) use ($searchTerm) {
                $q->where('product_title', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhere('product_description', 'LIKE', '%' . $searchTerm . '%')
                    ->orWhereHas('category', function ($query) use ($searchTerm) {
                        $query->where('name', 'LIKE', '%' . $searchTerm . '%');
                    });
            });
        }

        // Filter harga produk
        if ($request->has('min_price') && $request->min_price != '') {
            $productQuery->where('product_prices', '>=', $request->min_price);
        }

        if ($request->has('max_price') && $request->max_price != '') {
            $productQuery->where('product_prices', '<=', $request->max_price);
        }

        // Hitung total items sebelum pagination
        $totalItems = $productQuery->count();

        // Load relasi yang diperlukan dan paginate
        $products = $productQuery->with([
            'images',
            'category',
            'specifications'
        ])->paginate(12)->onEachSide(3)->withQueryString();

        // Data untuk view
        $selectedCategory = $request->get('category', '');

        return view('home.best-selling', compact(
            'products',
            'subCategories',
            'parentCategories',
            'totalItems',
            'selectedCategory'
        ));
    }

    // Halaman Portfolio Detail
    public function portfolioDetail($id)
    {
        // Cek apakah ID adalah numeric atau string
        if (is_numeric($id)) {
            // Ambil produk berdasarkan ID
            $product = Product::with([
                'category',
                'specifications',
                'images',
                'materials',
                'spareparts',
                'orders' => function($query) {
                    $query->where('order_status', 'diterima');
                }
            ])->findOrFail($id);
        } else {
            // Jika ID adalah string, cari produk "Mesin Pembelah Pinang" atau produk dengan banyak images
            $product = Product::where('product_title', 'LIKE', '%Pembelah Pinang%')
                ->orWhere('product_title', 'LIKE', '%pembelah pinang%')
                ->with([
                    'category',
                    'specifications',
                    'images',
                    'materials',
                    'spareparts',
                    'orders' => function($query) {
                        $query->where('order_status', 'diterima');
                    }
                ])->first();
            
            // Jika tidak ditemukan, ambil produk dengan banyak images
            if (!$product) {
                $product = Product::withCount('images')
                    ->where('is_active', true)
                    ->with([
                        'category',
                        'specifications',
                        'images',
                        'materials',
                        'spareparts',
                        'orders' => function($query) {
                            $query->where('order_status', 'diterima');
                        }
                    ])
                    ->orderBy('images_count', 'desc')
                    ->first();
            }
            
            if (!$product) {
                abort(404, 'Produk tidak ditemukan');
            }
        }

        // Hitung jumlah unit yang sudah diproduksi
        $totalProduced = $product->orders()->where('order_status', 'diterima')->count();

        // Ambil produk terkait (dari kategori yang sama, exclude produk saat ini)
        $relatedProducts = Product::where('is_active', true)
            ->where('id', '!=', $product->id)
            ->where(function($query) use ($product) {
                if ($product->category_id) {
                    $query->where('category_id', $product->category_id);
                }
            })
            ->with(['category', 'images'])
            ->take(4)
            ->get();

        // Jika tidak ada produk terkait dari kategori yang sama, ambil produk terbaru
        if ($relatedProducts->isEmpty()) {
            $relatedProducts = Product::where('is_active', true)
                ->where('id', '!=', $product->id)
                ->with(['category', 'images'])
                ->latest()
                ->take(4)
                ->get();
        }

        return view('home.portfolio-detail', compact('product', 'totalProduced', 'relatedProducts'));
    }
}
