<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Product;
use App\Models\Production;
use App\Models\User;
use App\Models\Setting;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Midtrans\Config as MidtransConfig;
use Midtrans\Snap;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class OrderController extends Controller
{
    /**
     * 1. Tampilkan form checkout
     */
    public function create(Product $product)
    {
        // Pastikan user sudah login untuk checkout
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu untuk melakukan pemesanan.');
        }
        
        return view('order.checkout', compact('product'));
    }

    /**
     * 2. Simpan order baru (checkout)
     */
    public function store(Request $request, Product $product)
    {
        // Pastikan user sudah login
        if (!Auth::check()) {
            return response()->json([
                'error' => true,
                'message' => 'Silakan login terlebih dahulu untuk melakukan pemesanan.',
            ], 401);
        }

        $request->validate([
            'customer_name'  => 'required|string|max:255',
            'customer_email' => 'required|email',
            'phone_number'   => 'required|string|max:20',
            'address'        => 'required|string',
            'quantity'       => 'required|integer|min:1',
            'payment_method' => 'required|string|in:transfer,midtrans,cod,dp',
        ]);

        // Validasi: Email harus sama dengan email user yang login (kecuali admin)
        $user = Auth::user();
        if ($user->user_type !== 'admin' && strtolower(trim($request->customer_email)) !== strtolower(trim($user->email))) {
            return response()->json([
                'error' => true,
                'message' => 'Email harus sama dengan email akun Anda yang sedang login.',
            ], 422);
        }

        DB::beginTransaction();
        try {
            $quantity = $request->quantity;
            $price = $product->product_prices;
            $total = $price * $quantity;

            // Untuk Midtrans: Hapus SEMUA order pending dari user yang sama dengan produk yang sama
            // Ini untuk mencegah duplikasi jika user batalkan pembayaran dan checkout lagi
            if ($request->payment_method === 'midtrans' && Auth::check()) {
                $oldPendingOrders = Order::where('user_id', Auth::id())
                    ->where('payment_method', 'midtrans')
                    ->where('product_id', $product->id) // Hanya order dengan produk yang sama
                    ->where(function($query) {
                        // Order yang belum ada update dari Midtrans (NULL atau pending)
                        $query->whereNull('midtrans_status')
                              ->orWhere('midtrans_status', 'pending');
                    })
                    ->where(function($query) {
                        // Order yang belum settlement (belum berhasil bayar)
                        $query->where('status', '!=', 'selesai')
                              ->orWhereNull('status');
                    })
                    ->get(); // Hapus semua, tidak peduli umurnya
                
                foreach ($oldPendingOrders as $oldOrder) {
                    // Hapus production jika ada
                    if ($oldOrder->production) {
                        $oldOrder->production->delete();
                    }
                    // Hapus receivable jika ada
                    if ($oldOrder->receivable) {
                        $oldOrder->receivable->delete();
                    }
                    // Hapus order
                    $oldOrder->delete();
                    
                    Log::info('Old pending Midtrans order deleted before creating new one', [
                        'old_order_id' => $oldOrder->id,
                        'user_id' => Auth::id(),
                        'product_id' => $product->id,
                        'midtrans_status' => $oldOrder->midtrans_status,
                    ]);
                }
            }

            // Status awal order
            $status = in_array($request->payment_method, ['cod']) ? 'diproses' : ($request->payment_method === 'dp' ? 'menunggu' : 'menunggu');

            $order = Order::create([
                'customer_name'     => $request->customer_name,
                'customer_email'    => $request->customer_email,
                'phone_number'      => $request->phone_number,
                'address'           => $request->address,
                'user_id'           => Auth::id(),
                'product_id'        => $product->id,
                'quantity'          => $quantity,
                'price'             => $price,
                'total_price'       => $total,
                'payment_method'    => $request->payment_method,
                'dp_percentage'     => $request->payment_method === 'dp' ? 30.00 : null, // Default 30% untuk DP
                'status'            => $status,
                'order_status'      => 'menunggu', // Default order status
                'production_status' => 'menunggu',
                'is_read'           => false, // Belum dibaca admin
                'order_type'        => 'beli_langsung', // Jenis pesanan untuk checkout langsung
            ]);

            DB::commit();

            // Kirim notifikasi untuk pesanan baru
            try {
                $notificationService = new NotificationService();
                $notificationService->notifyNewOrder($order, true);
            } catch (\Exception $e) {
                Log::error('Failed to send order notification: ' . $e->getMessage());
            }

            // Midtrans
            if ($request->payment_method === 'midtrans') {
                try {
                    // Order baru, bukan retry, jadi isRetry = false (default)
                    $snapResponse = $this->generateSnapToken($order, $product, false);
                    return response()->json([
                        'snap_token' => $snapResponse['snap_token'],
                        'order_id'   => $order->id,
                    ]);
                } catch (\Exception $e) {
                    // Rollback transaction jika error
                    DB::rollBack();
                    Log::error('Error generating Midtrans token: ' . $e->getMessage());
                    return response()->json([
                        'error' => true,
                        'message' => 'Gagal memproses pembayaran. ' . $e->getMessage(),
                    ], 500);
                }
            }

            // Redirect untuk metode lain
            $redirectUrl = match ($request->payment_method) {
                'transfer' => route('orders.confirm.form', $order),
                'cod' => route('orders.success', $order),
                'dp' => route('orders.success', $order),
                default => route('all'),
            };

            $message = match ($request->payment_method) {
                'transfer' => 'Pesanan berhasil dibuat. Silakan upload bukti transfer.',
                'cod' => 'Pesanan siap! Bayar saat kurir tiba di alamat Anda.',
                'dp' => 'Pesanan DP berhasil dibuat. Silakan lakukan pembayaran DP sesuai instruksi.',
                default => 'Pesanan berhasil dibuat.',
            };

            return response()->json([
                'success' => true,
                'redirect_url' => $redirectUrl,
                'order_id' => $order->id,
                'message' => $message,
            ]);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json([
                'error' => true,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Helper: Generate Snap Token
     * 
     * @param Order $order
     * @param Product $product
     * @param bool $isRetry Apakah ini retry payment (gunakan order_id unik)
     * @throws \Exception jika terjadi error saat generate token
     */
    private function generateSnapToken(Order $order, Product $product, $isRetry = false)
    {
        try {
            // Validasi konfigurasi Midtrans
            $serverKey = config('midtrans.server_key');
            $clientKey = config('midtrans.client_key');
            
            if (empty($serverKey) || empty($clientKey)) {
                throw new \Exception('Midtrans Server Key atau Client Key belum dikonfigurasi. Silakan cek file .env');
            }

            // Setup Midtrans Config
            MidtransConfig::$serverKey = $serverKey;
            MidtransConfig::$isSanitized = true;
            MidtransConfig::$is3ds = true;
            MidtransConfig::$isProduction = config('midtrans.is_production', false);

            // Validasi gross_amount
            if ($order->total_price <= 0) {
                throw new \Exception('Total harga tidak valid: ' . $order->total_price);
            }

            // Untuk retry, gunakan order_id yang unik dengan timestamp
            // Midtrans requirement: 
            // - Max 50 karakter
            // - Hanya alphanumeric, dash (-), underscore (_)
            // - Harus unique untuk setiap transaksi
            $midtransOrderId = (string) $order->id;
            if ($isRetry) {
                // Gunakan uniqid dengan prefix untuk memastikan unik setiap kali retry
                // Format: {order_id}_r{uniqid} (contoh: 75_r67890abc123)
                $uniqueId = substr(uniqid('', true), 0, 13); // Ambil 13 karakter dari uniqid
                $midtransOrderId = $order->id . '_r' . str_replace('.', '', $uniqueId); // Hapus titik dari uniqid
                
                // Midtrans membatasi order_id max 50 karakter
                if (strlen($midtransOrderId) > 50) {
                    // Jika terlalu panjang, gunakan hash yang lebih pendek
                    $hash = substr(md5($order->id . time() . uniqid()), 0, 10);
                    $midtransOrderId = $order->id . '_r' . $hash;
                    
                    // Jika masih terlalu panjang, potong order_id
                    if (strlen($midtransOrderId) > 50) {
                        $maxOrderIdLength = 50 - strlen('_r' . $hash);
                        if ($maxOrderIdLength > 0) {
                            $midtransOrderId = substr((string)$order->id, 0, $maxOrderIdLength) . '_r' . $hash;
                        } else {
                            // Fallback: gunakan hash saja jika order_id terlalu panjang
                            $midtransOrderId = 'r' . substr(md5($order->id . time() . uniqid()), 0, 49);
                        }
                    }
                }
                
                // Pastikan hanya mengandung karakter yang diizinkan Midtrans (alphanumeric, dash, underscore)
                $midtransOrderId = preg_replace('/[^a-zA-Z0-9_-]/', '', $midtransOrderId);
                
                // Pastikan tidak kosong
                if (empty($midtransOrderId)) {
                    $midtransOrderId = 'r' . substr(md5($order->id . time() . uniqid()), 0, 49);
                }
            }

            // Validasi format order_id sebelum kirim ke Midtrans
            if (empty($midtransOrderId) || strlen($midtransOrderId) > 50) {
                throw new \Exception('Order ID tidak valid: ' . $midtransOrderId . ' (length: ' . strlen($midtransOrderId) . ')');
            }

            $params = [
                'transaction_details' => [
                    'order_id' => $midtransOrderId, // Order ID unik untuk retry
                    'gross_amount' => (int) $order->total_price, // Pastikan integer
                ],
                'customer_details' => [
                    'first_name' => $order->customer_name,
                    'last_name' => '', // Optional
                    'email' => $order->customer_email,
                    'phone' => $order->phone_number,
                ],
                'enabled_payments' => ['qris', 'gopay', 'shopeepay'], // QRIS diutamakan
                'finish_redirect_url' => route('orders.success', $order),
                'callbacks' => [
                    'finish' => route('orders.success', $order),
                ],
            ];

            // Log parameter sebelum kirim ke Midtrans
            Log::info('Generating Midtrans Snap Token', [
                'order_id' => $order->id,
                'midtrans_order_id' => $midtransOrderId,
                'is_retry' => $isRetry,
                'gross_amount' => $order->total_price,
                'order_id_length' => strlen($midtransOrderId),
            ]);

            // Generate Snap Token
            $snapToken = Snap::getSnapToken($params);

            if (empty($snapToken)) {
                throw new \Exception('Gagal generate Snap Token dari Midtrans');
            }

            // Update order status dan simpan midtrans_order_id untuk retry
            $updateData = ['midtrans_status' => 'pending'];
            if ($isRetry) {
                $updateData['midtrans_transaction_id'] = $midtransOrderId; // Simpan untuk tracking
            }
            $order->update($updateData);

            Log::info('Midtrans Snap Token generated successfully', [
                'order_id' => $order->id,
                'midtrans_order_id' => $midtransOrderId,
                'is_retry' => $isRetry,
                'gross_amount' => $order->total_price,
            ]);

            return [
                'snap_token' => $snapToken,
                'midtrans_order_id' => $midtransOrderId, // Return untuk tracking
            ];
            
        } catch (\Midtrans\Exception $e) {
            Log::error('Midtrans API Error: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'error_code' => $e->getCode(),
            ]);
            throw new \Exception('Error Midtrans: ' . $e->getMessage());
        } catch (\Exception $e) {
            Log::error('Error generating Snap Token: ' . $e->getMessage(), [
                'order_id' => $order->id,
            ]);
            throw $e;
        }
    }

    /**
     * 3a. Form konfirmasi transfer
     */
    public function showConfirmForm(Order $order)
    {
        // Route model binding sudah handle authorization di Order model
        // Tapi tetap tambahkan double check untuk keamanan ekstra
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu.');
        }

        // Double check authorization (case-insensitive untuk production)
        $user = Auth::user();
        $hasAccess = false;
        
        if ($order->user_id !== null) {
            $hasAccess = ((int)$order->user_id === (int)$user->id);
        } else {
            // Case-insensitive email comparison untuk production
            $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
        }
        
        if (!$hasAccess) {
            return redirect()->route('my.orders.index')->with('error', 'Akses ditolak. Anda tidak memiliki akses ke order ini.');
        }

        // Validasi: hanya untuk transfer bank atau DP
        if (!in_array($order->payment_method, ['transfer', 'dp'])) {
            return redirect()->back()->with('error', 'Konfirmasi hanya untuk transfer bank atau DP.');
        }
        return view('order.confirm-form', compact('order'));
    }

    /**
     * 3b. Konfirmasi transfer
     */
    public function confirm(Request $request, Order $order)
    {
        // Route model binding sudah handle authorization di Order model
        // Tapi tetap tambahkan double check untuk keamanan ekstra
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu.');
        }

        // Double check authorization (case-insensitive untuk production)
        $user = Auth::user();
        $hasAccess = false;
        
        if ($order->user_id !== null) {
            $hasAccess = ((int)$order->user_id === (int)$user->id);
        } else {
            // Case-insensitive email comparison untuk production
            $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
        }
        
        if (!$hasAccess) {
            return redirect()->route('my.orders.index')->with('error', 'Akses ditolak. Anda tidak memiliki akses ke order ini.');
        }

        // Validasi: hanya untuk transfer bank atau DP
        if (!in_array($order->payment_method, ['transfer', 'dp'])) {
            return redirect()->back()->with('error', 'Konfirmasi hanya untuk transfer bank atau DP.');
        }

        $request->validate([
            'proof_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'payment_type' => 'nullable|string|in:transfer,cash', // Untuk DP: transfer atau cash
        ]);

        // Hapus file lama jika ada
        if ($order->proof_file && Storage::disk('uploads')->exists($order->proof_file)) {
            Storage::disk('uploads')->delete($order->proof_file);
        }

        // Simpan file baru
        $path = $request->file('proof_file')->store('proofs', 'uploads');

        // Update order dan refresh untuk memastikan data terbaru
        $updateData = [
            'proof_file' => $path,
        ];

        // Untuk transfer bank dan DP, ubah status menjadi menunggu verifikasi
        // Ini membantu admin mengetahui ada bukti pembayaran yang perlu diverifikasi
        // Setelah admin verifikasi, status akan diupdate sesuai kebutuhan:
        // - Transfer: bisa langsung "diproses" atau "selesai" (jika lunas)
        // - DP: bisa "diproses" untuk mulai produksi (karena DP sudah dibayar 30%)
        if (in_array($order->payment_method, ['transfer', 'dp'])) {
            $updateData['status'] = 'menunggu_verifikasi';
        }

        $order->update($updateData);
        
        // Refresh order untuk memastikan data terbaru
        $order->refresh();
        
        // Kirim notifikasi ke admin bahwa bukti pembayaran telah diupload
        try {
            $notificationService = new NotificationService();
            $notificationService->notifyPaymentProofUploaded($order);
        } catch (\Exception $e) {
            Log::error('Failed to send payment proof uploaded notification: ' . $e->getMessage());
        }
        
        // Production hanya akan dibuat setelah order disetujui dan pembayaran diverifikasi
        // Lihat updatePaymentStatus dan updateStatus untuk implementasi

        // Pesan sukses berbeda untuk transfer dan DP
        $successMessage = $order->payment_method === 'dp' 
            ? 'Bukti pembayaran DP berhasil dikirim. Admin akan verifikasi segera. Setelah DP dikonfirmasi, proses produksi akan dimulai.'
            : 'Bukti transfer berhasil dikirim. Admin akan verifikasi segera.';

        // Redirect kembali ke halaman detail order
        return redirect()->route('my.orders.show', $order)
            ->with('success', $successMessage);
    }

    /**
     * 4. Handle Midtrans notification (Webhook)
     * 
     * URL: /midtrans/notification
     * Method: POST
     * 
     * Midtrans akan mengirim notifikasi ke URL ini setelah payment status berubah.
     * Pastikan URL ini sudah di-set di Midtrans Dashboard → Settings → Configuration → Payment Notification URL
     * 
     * Untuk test dari Midtrans Dashboard, endpoint ini akan return 200 OK meskipun data tidak lengkap
     * (karena test notification tidak mengirim data lengkap seperti production)
     */
    public function handleNotification(Request $request)
    {
        try {
            // Log semua request untuk debugging
            Log::info('Midtrans notification received', [
                'headers' => $request->headers->all(),
                'body' => $request->all(),
                'method' => $request->method(),
                'url' => $request->fullUrl(),
            ]);

            // Handle test notification dari Midtrans Dashboard
            // Test notification biasanya tidak mengirim data lengkap
            if ($request->has('test') || empty($request->order_id)) {
                Log::info('Midtrans test notification received');
                return response()->json([
                    'status' => 'ok',
                    'message' => 'Webhook endpoint is accessible',
                    'test' => true
                ], 200);
            }

            $serverKey = config('midtrans.server_key');
            
            if (empty($serverKey)) {
                Log::error('Midtrans Server Key tidak dikonfigurasi');
                return response()->json(['message' => 'Server configuration error'], 500);
            }
            
            // Ambil data dari request
            $orderId = $request->order_id;
            $statusCode = $request->status_code;
            $grossAmount = $request->gross_amount;
            $transactionStatus = $request->transaction_status;
            $transactionId = $request->transaction_id;
            $paymentType = $request->payment_type;
            
            // Log detail notification
            Log::info('Midtrans notification details', [
                'order_id' => $orderId,
                'transaction_id' => $transactionId,
                'transaction_status' => $transactionStatus,
                'payment_type' => $paymentType,
                'status_code' => $statusCode,
                'gross_amount' => $grossAmount,
                'all_request_data' => $request->all(), // Log semua data untuk debugging
            ]);
            
            if (empty($orderId)) {
                Log::warning('Midtrans notification: order_id kosong', [
                    'all_data' => $request->all(),
                ]);
                return response()->json([
                    'status' => 'ok',
                    'message' => 'Notification received but order_id is missing',
                ], 200);
            }

            // Cek apakah transaction_status ada, jika tidak ada berarti mungkin test notification
            if (empty($transactionStatus)) {
                Log::warning('Midtrans notification: transaction_status kosong', [
                    'order_id' => $orderId,
                    'all_data' => $request->all(),
                ]);
                // Tetap cari order dan cek apakah perlu dihapus jika pending terlalu lama
                $order = Order::find($orderId);
                if ($order && $order->payment_method === 'midtrans' && empty($order->midtrans_status)) {
                    // Jika order sudah lebih dari 1 jam dan masih pending, hapus
                    if ($order->created_at->diffInHours(now()) >= 1) {
                        $order->delete();
                        Log::info('Midtrans order deleted (pending too long, no status update)', [
                            'order_id' => $orderId,
                        ]);
                    }
                }
                return response()->json([
                    'status' => 'ok',
                    'message' => 'Notification received but transaction_status is missing',
                ], 200);
            }

            // Verifikasi signature untuk keamanan (optional untuk sandbox, wajib untuk production)
            $isProduction = config('midtrans.is_production', false);
            if ($isProduction && !empty($statusCode) && !empty($grossAmount)) {
                $expectedSignature = hash("sha512", $orderId . $statusCode . $grossAmount . $serverKey);
                if (!empty($request->signature_key) && $expectedSignature !== $request->signature_key) {
                    Log::error('Midtrans signature mismatch', [
                        'expected' => $expectedSignature,
                        'received' => $request->signature_key,
                        'order_id' => $orderId,
                    ]);
                    return response()->json(['message' => 'Invalid signature'], 403);
                }
            } else {
                Log::info('Midtrans signature verification skipped (sandbox mode or incomplete data)');
            }

            // Extract order_id asli jika ini retry payment
            // Format baru: {order_id}_r{timestamp} atau {order_id}_r{hash}
            $originalOrderId = $orderId;
            if (strpos($orderId, '_r') !== false) {
                // Format: {order_id}_r{timestamp} atau {order_id}_r{hash}
                // Coba extract dengan regex untuk angka di awal
                if (preg_match('/^(\d+)_r/', $orderId, $matches)) {
                    $originalOrderId = $matches[1];
                } elseif (strpos($orderId, '_retry_') !== false) {
                    // Format lama: {order_id}_retry_{timestamp}
                    $parts = explode('_retry_', $orderId);
                    $originalOrderId = $parts[0];
                }
            }

            // Cari order berdasarkan order_id asli
            $order = Order::find($originalOrderId);
            if (!$order) {
                Log::warning('Midtrans order not found: ' . $orderId . ' (original: ' . $originalOrderId . ') - mungkin sudah dihapus atau belum dibuat');
                // Return 200 untuk menghindari Midtrans retry terus menerus
                return response()->json([
                    'success' => true,
                    'message' => 'Order not found (might be already deleted)',
                ], 200);
            }

            // Log untuk tracking retry payment
            if ($orderId !== $originalOrderId) {
                Log::info('Midtrans retry payment notification received', [
                    'midtrans_order_id' => $orderId,
                    'original_order_id' => $originalOrderId,
                    'order_id' => $order->id,
                ]);
            }

            // Validasi: hanya proses order dengan payment_method = midtrans
            if ($order->payment_method !== 'midtrans') {
                Log::warning('Midtrans notification for non-midtrans order', [
                    'order_id' => $orderId,
                    'payment_method' => $order->payment_method,
                ]);
                return response()->json([
                    'success' => true,
                    'message' => 'Order is not using Midtrans payment',
                ], 200);
            }

            // Map transaction status ke status pembayaran sistem
            $status = match ($transactionStatus) {
                'settlement' => 'selesai',
                'pending' => 'diproses',
                'deny', 'cancel', 'expire' => 'gagal',
                default => $order->status ?? 'diproses', // Fallback ke status existing jika tidak dikenal
            };

            // Handle VA number (hanya untuk Virtual Account)
            $vaNumber = null;
            if (!empty($request->va_numbers) && is_array($request->va_numbers) && isset($request->va_numbers[0]['va_number'])) {
                $vaNumber = $request->va_numbers[0]['va_number'];
            }

            // Update order dengan transaction data
            DB::beginTransaction();
            try {
                // Prepare update data
                $updateData = [
                    'midtrans_status' => $transactionStatus,
                    'status' => $status,
                ];

                // Update transaction_id jika ada (selalu update, jangan fallback ke existing)
                if (!empty($transactionId)) {
                    $updateData['midtrans_transaction_id'] = $transactionId;
                }

                // Update payment_type jika ada
                if (!empty($paymentType)) {
                    $updateData['midtrans_payment_type'] = $paymentType;
                }

                // Update VA number jika ada (hanya untuk Virtual Account)
                if (!empty($vaNumber)) {
                    $updateData['midtrans_va_number'] = $vaNumber;
                }

                // Jika pembayaran dibatalkan/gagal: hapus order
                if (in_array($transactionStatus, ['cancel', 'expire', 'deny'])) {
                    // Hapus order yang gagal dari database
                    $orderIdToDelete = $order->id;
                    
                    // Hapus production jika ada
                    if ($order->production) {
                        $order->production->delete();
                    }
                    
                    // Hapus receivable jika ada
                    if ($order->receivable) {
                        $order->receivable->delete();
                    }
                    
                    // Hapus order
                    $order->delete();
                    
                    Log::info('Midtrans payment failed, order deleted', [
                        'order_id' => $orderIdToDelete,
                        'transaction_status' => $transactionStatus,
                    ]);
                    
                    DB::commit();
                    
                    return response()->json([
                        'success' => true,
                        'message' => 'Payment failed, order deleted',
                    ]);
                }

                // Jika status pending dan sudah lebih dari 1 jam: hapus order (timeout)
                if ($transactionStatus === 'pending' && $order->created_at->diffInHours(now()) >= 1) {
                    $orderIdToDelete = $order->id;
                    
                    // Hapus production jika ada
                    if ($order->production) {
                        $order->production->delete();
                    }
                    
                    // Hapus receivable jika ada
                    if ($order->receivable) {
                        $order->receivable->delete();
                    }
                    
                    // Hapus order
                    $order->delete();
                    
                    Log::info('Midtrans order deleted (pending timeout > 1 hour)', [
                        'order_id' => $orderIdToDelete,
                        'transaction_status' => $transactionStatus,
                    ]);
                    
                    DB::commit();
                    
                    return response()->json([
                        'success' => true,
                        'message' => 'Order deleted due to pending timeout',
                    ]);
                }

                // Update order dengan transaction data
                $order->update($updateData);

                // Jika settlement (pembayaran berhasil), update order_status dan buat production
                if ($transactionStatus === 'settlement') {
                    $oldOrderStatus = $order->order_status;
                    
                    // Update order_status otomatis menjadi 'diterima' untuk Midtrans
                    // Karena pembayaran sudah selesai, order langsung diterima
                    if ($order->order_status === 'menunggu') {
                        $order->updateQuietly([
                            'order_status' => 'diterima',
                        ]);
                    }

                    // Buat produksi otomatis jika belum ada
                    Production::firstOrCreate(
                        ['order_id' => $order->id],
                        [
                            'product_id' => $order->product_id,
                            'quantity' => $order->quantity,
                            'start_date' => now(),
                            'status' => 'menunggu', // Tetap menunggu, tidak langsung dalam_proses
                            'total_material_cost' => 0,
                            'total_sparepart_cost' => 0,
                            'total_production_cost' => 0,
                        ]
                    );
                    
                    // Update order production_status
                    $order->updateQuietly([
                        'production_status' => 'menunggu',
                    ]);

                    // Refresh order untuk mendapatkan data terbaru
                    $order->refresh();

                    // Kirim notifikasi pembayaran berhasil ke user dan admin
                    try {
                        $notificationService = new NotificationService();
                        $notificationService->notifyPaymentReceived($order);
                        
                        // Kirim notifikasi status order berubah jika status berubah
                        if ($oldOrderStatus !== $order->order_status) {
                            $notificationService->notifyOrderStatusChanged($order, $oldOrderStatus, $order->order_status);
                        }
                    } catch (\Exception $e) {
                        Log::error('Failed to send Midtrans settlement notification: ' . $e->getMessage());
                    }

                    Log::info('Midtrans settlement: Order status updated and production created', [
                        'order_id' => $order->id,
                        'order_status' => $order->order_status,
                        'payment_status' => $status,
                    ]);
                }

                DB::commit();
                
                Log::info('Midtrans notification processed successfully', [
                    'order_id' => $order->id,
                    'transaction_id' => $transactionId,
                    'transaction_status' => $transactionStatus,
                    'payment_type' => $paymentType,
                    'payment_status' => $status,
                    'order_status' => $order->order_status,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Notification processed successfully',
                    'order_id' => $order->id,
                    'status' => $status,
                ]);
                
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error processing Midtrans notification: ' . $e->getMessage(), [
                    'order_id' => $orderId,
                    'transaction_id' => $transactionId,
                    'transaction_status' => $transactionStatus,
                    'trace' => $e->getTraceAsString(),
                ]);
                return response()->json(['message' => 'Error processing notification'], 500);
            }
            
        } catch (\Exception $e) {
            Log::error('Fatal error in Midtrans notification handler: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json(['message' => 'Internal server error'], 500);
        }
    }

    /**
     * 5. Halaman success
     */
    public function showSuccess(Order $order)
    {
        // Route model binding sudah handle authorization di Order model
        // Tapi tetap tambahkan double check untuk keamanan ekstra
        if (Auth::check()) {
            $user = Auth::user();
            $hasAccess = false;
            
            if ($order->user_id !== null) {
                $hasAccess = ((int)$order->user_id === (int)$user->id);
            } else {
                // Case-insensitive email comparison untuk production
                $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
            }
            
            if (!$hasAccess) {
                return redirect()->route('all')->with('error', 'Akses ditolak. Anda tidak memiliki akses ke order ini.');
            }
        }
        return view('order.success', compact('order'));
    }

    /**
     * 6. 
     *  list orders - Gabungan Order dan Sale
     */
    public function adminIndex(Request $request)
    {
        $search = $request->input('search');
        $tab = $request->input('tab', 'all'); // all, orders (sales sudah dipisah ke menu terpisah)

        // Query untuk Order
        $ordersQuery = Order::with(['user', 'product', 'receivable', 'sale']);

        // Filter berdasarkan tab - HARUS DULUAN sebelum where clause lain
        if ($tab === 'orders') {
            // Hanya order yang belum jadi sale
            $ordersQuery->whereDoesntHave('sale');
        }
        // Tab 'sales' sudah dihapus karena sudah ada menu terpisah di sidebar

        // Filter berdasarkan search - perbaiki dengan grouping yang benar
        if ($search && trim($search) !== '') {
            $searchTerm = trim($search);
            $ordersQuery->where(function ($query) use ($searchTerm) {
                // Cek apakah search adalah angka (kemungkinan order ID)
                if (is_numeric($searchTerm)) {
                    $query->where('id', $searchTerm)
                        ->orWhere('customer_name', 'like', "%{$searchTerm}%")
                        ->orWhere('customer_email', 'like', "%{$searchTerm}%")
                        ->orWhere('phone_number', 'like', "%{$searchTerm}%")
                        ->orWhereHas('product', function($q) use ($searchTerm) {
                            $q->where('product_title', 'like', "%{$searchTerm}%");
                        })
                        ->orWhere('status', 'like', "%{$searchTerm}%");
                } else {
                    // Jika bukan angka, cari di field text biasa
                    $query->where('customer_name', 'like', "%{$searchTerm}%")
                        ->orWhere('customer_email', 'like', "%{$searchTerm}%")
                        ->orWhere('phone_number', 'like', "%{$searchTerm}%")
                        ->orWhereHas('product', function($q) use ($searchTerm) {
                            $q->where('product_title', 'like', "%{$searchTerm}%");
                        })
                        ->orWhere('status', 'like', "%{$searchTerm}%");
                }
            });
        }

        // Filter berdasarkan order_status
        if ($request->filled('order_status')) {
            $ordersQuery->where('order_status', $request->order_status);
        }

        // Filter berdasarkan payment_status (status pembayaran)
        if ($request->filled('payment_status')) {
            $ordersQuery->where('status', $request->payment_status);
        }

        // Filter berdasarkan production_status
        if ($request->filled('production_status')) {
            $ordersQuery->where('production_status', $request->production_status);
        }

        // Filter berdasarkan payment_method
        if ($request->filled('payment_method')) {
            $ordersQuery->where('payment_method', $request->payment_method);
        }

        // Filter berdasarkan order_type
        if ($request->filled('order_type')) {
            $ordersQuery->where('order_type', $request->order_type);
        }

        // Urutkan unread (is_read = false/null) di atas terlebih dahulu, lalu latest created_at
        $itemsPerPage = (int) Setting::get('items_per_page', 10);
        $orders = $ordersQuery->orderByRaw('COALESCE(is_read, 0) ASC') // false/null (0) di atas
            ->orderBy('created_at', 'desc')
            ->paginate($itemsPerPage)
            ->withQueryString();

        // Ambil data products untuk form create
        $products = Product::where('is_active', true)->get();

        // Statistik untuk tab
        $stats = [
            'total_orders' => Order::count(),
            'orders_with_sales' => Order::whereHas('sale')->count(),
            'orders_without_sales' => Order::whereDoesntHave('sale')->count(),
        ];

        return view('admin.master.orders.index', compact('orders', 'search', 'products', 'tab', 'stats'));
    }

    /**
     * 7. Admin show order
     */
    public function adminShow(Order $order)
    {
        $order->load(['user', 'product']);
        
        // Tandai pesanan sebagai sudah dibaca saat admin melihat detail
        if (!$order->is_read) {
            $order->update(['is_read' => true]);
        }
        
        return view('admin.master.orders.show', compact('order'));
    }

    /**
     * 8. Admin store order
     */
    public function adminStore(Request $request)
    {
        $request->validate([
            'customer_name'  => 'required|string|max:255',
            'customer_email' => 'required|email',
            'phone_number'   => 'required|string|max:20',
            'address'        => 'required|string',
            'product_id'     => 'required|exists:products,id',
            'quantity'       => 'required|integer|min:1',
            'payment_method' => 'required|string|in:transfer,cod,dp',
            'order_status'   => 'required|string|in:menunggu,diterima,ditolak',
        ]);

        DB::beginTransaction();
        try {
            $product = Product::findOrFail($request->product_id);
            $quantity = $request->quantity;
            $price = $product->product_prices;
            $total = $price * $quantity;

            $order = Order::create([
                'customer_name'     => $request->customer_name,
                'customer_email'    => $request->customer_email,
                'phone_number'      => $request->phone_number,
                'address'           => $request->address,
                'user_id'           => null, // Manual order tidak perlu user_id
                'product_id'        => $product->id,
                'quantity'          => $quantity,
                'price'             => $price,
                'total_price'       => $total,
                'payment_method'    => $request->payment_method,
                'dp_percentage'     => $request->payment_method === 'dp' ? 30.00 : null, // Default 30% untuk DP
                'status'            => in_array($request->payment_method, ['cod']) ? 'diproses' : ($request->payment_method === 'dp' ? 'menunggu' : 'menunggu'), // Status pembayaran
                'order_status'      => $request->order_status, // Status pesanan
                'production_status' => 'menunggu',
                'is_read'           => false, // Belum dibaca admin
                'order_type'        => 'beli_langsung', // Jenis pesanan untuk pesanan yang dibuat admin
            ]);

            // Buat produksi otomatis jika order_status diterima
            // CATATAN: Production dibuat dengan status 'menunggu', tidak langsung 'dalam_proses'
            // Status akan berubah ke 'dalam_proses' saat teknisi mulai produksi
            if ($request->order_status === 'diterima') {
                Production::firstOrCreate(
                    ['order_id' => $order->id],
                    [
                        'product_id' => $order->product_id,
                        'quantity' => $order->quantity,
                        'start_date' => now(),
                        'status' => 'menunggu', // Tetap menunggu, tidak langsung dalam_proses
                        'total_material_cost' => 0,
                        'total_sparepart_cost' => 0,
                        'total_production_cost' => 0,
                    ]
                );

                $order->updateQuietly([
                    'production_status' => 'menunggu',
                ]);
            }

            // Receivable akan dibuat otomatis di Model Order::booted()

            DB::commit();

            // Kirim notifikasi untuk pesanan baru
            try {
                $notificationService = new NotificationService();
                $notificationService->notifyNewOrder($order, true);
            } catch (\Exception $e) {
                Log::error('Failed to send order notification: ' . $e->getMessage());
            }

            return redirect()->route('admin.master.orders.index')
                ->with('success', "Pesanan manual berhasil dibuat dengan ID #{$order->id}");
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * 10. Update status order (admin)
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|string|in:menunggu,diterima,ditolak',
        ]);

        $order = Order::findOrFail($id);
        $oldStatus = $order->order_status;
        
        // Validasi: Jangan ubah status jika order sudah ditolak
        if ($order->order_status === 'ditolak' && $request->status !== 'ditolak') {
            return back()->withErrors(['status' => 'Order yang sudah ditolak tidak bisa diubah statusnya.']);
        }

        $order->update(['order_status' => $request->status]);

        // Kirim notifikasi status order berubah
        // Catatan: notifyOrderStatusChanged sudah menangani pesan khusus untuk status 'ditolak'
        // jadi tidak perlu memanggil notifyOrderRejected lagi (menghindari duplikasi)
        try {
            $notificationService = new NotificationService();
            $notificationService->notifyOrderStatusChanged($order, $oldStatus, $request->status);
        } catch (\Exception $e) {
            Log::error('Failed to send order status notification: ' . $e->getMessage());
        }

        // Jika order diterima, buat produksi jika belum ada dan pembayaran sudah diproses/selesai
        // CATATAN: Production bisa dibuat meskipun pembayaran belum selesai (untuk planning)
        if ($request->status === 'diterima') {
            // Gunakan firstOrCreate untuk mencegah duplikasi
            Production::firstOrCreate(
                ['order_id' => $order->id],
                [
                    'product_id' => $order->product_id,
                    'quantity' => $order->quantity,
                    'start_date' => now(),
                    'status' => 'menunggu', // Menunggu ditugaskan ke teknisi dan planning
                    'total_material_cost' => 0,
                    'total_sparepart_cost' => 0,
                    'total_production_cost' => 0,
                ]
            );

            $order->updateQuietly([
                'production_status' => 'menunggu',
            ]);
        }

        return back()->with('success', "Status pesanan #{$order->id} berhasil diperbarui!");
    }

    /**
     * 11. Update payment status order (admin)
     */
    public function updatePaymentStatus(Request $request, $id)
    {
        $order = Order::findOrFail($id);
        $oldStatus = $order->status;

        // Validasi status pembayaran
        $request->validate([
            'payment_status' => 'required|in:menunggu,menunggu_verifikasi,diproses,selesai,gagal,dibatalkan',
        ]);

        // Update payment status (field yang benar)
        $order->update(['status' => $request->payment_status]);

        // Kirim notifikasi status pembayaran berubah
        try {
            $notificationService = new NotificationService();
            $notificationService->notifyPaymentStatusChanged($order, $oldStatus, $request->payment_status);
        } catch (\Exception $e) {
            Log::error('Failed to send payment status notification: ' . $e->getMessage());
        }

        // CATATAN: Production status TIDAK bergantung pada payment status
        // Production status hanya berubah melalui workflow produksi:
        // - menunggu → saat production dibuat dan belum ditugaskan
        // - menunggu → saat ditugaskan ke teknisi, planning dibuat/disetujui (tidak perlu tunggu payment)
        // - dalam_proses → saat teknisi mulai produksi (startProduction)
        // - selesai → saat completion approved
        
        // Jika pembayaran diproses/selesai dan order diterima, pastikan produksi ada (tapi tidak ubah status)
        if (in_array($request->payment_status, ['diproses', 'selesai']) && $order->order_status === 'diterima') {
            // Gunakan firstOrCreate untuk mencegah duplikasi (race condition)
            Production::firstOrCreate(
                ['order_id' => $order->id],
                [
                    'product_id' => $order->product_id,
                    'quantity' => $order->quantity,
                    'start_date' => now(),
                    'status' => 'menunggu', // Status tetap menunggu, tidak langsung dalam_proses
                    'total_material_cost' => 0,
                    'total_sparepart_cost' => 0,
                    'total_production_cost' => 0,
                ]
            );

            $order->updateQuietly([
                'production_status' => 'menunggu',
            ]);
            // Jika production sudah ada, TIDAK mengubah status (status mengikuti workflow produksi)

            // Update receivable jika payment_method = DP
            if ($order->payment_method === 'dp' && $order->receivable) {
                $receivable = $order->receivable;

                // Hanya update receivable jika status baru adalah "selesai" (pembayaran lunas)
                if ($request->payment_status === 'selesai' && $oldStatus !== 'selesai') {
                    // Kirim notifikasi pembayaran diterima
                    try {
                        $notificationService = new NotificationService();
                        $notificationService->notifyPaymentReceived($order);
                    } catch (\Exception $e) {
                        Log::error('Failed to send payment notification: ' . $e->getMessage());
                    }
                    
                    // Set pembayaran penuh (100%) karena status sudah selesai
                    $receivable->update([
                        'paid_amount' => $receivable->total_amount,
                    ]);

                    // Trigger update status
                    $receivable->updatePaymentStatus();
                } 
                // Jika status "diproses" (DP sudah dibayar dan diverifikasi)
                // CATATAN: Jangan otomatis set paid_amount di sini
                // Admin harus input manual di halaman Receivable sesuai jumlah yang benar-benar ditransfer
                // Karena customer mungkin transfer kurang atau lebih dari dp_percentage yang ditetapkan
                // Contoh: dp_percentage = 30%, tapi customer transfer hanya 15-20%
                // Admin akan input manual di halaman Receivable → Create Payment
                elseif ($request->payment_status === 'diproses' && $oldStatus !== 'diproses') {
                    // Jangan update paid_amount otomatis
                    // Biarkan admin input manual di halaman Receivable sesuai jumlah transfer yang sebenarnya
                    // paid_amount tetap 0 sampai admin input di Receivable
                }
            }
        }

        return back()->with('success', "Status pembayaran pesanan #{$order->id} berhasil diperbarui!");
    }

    /**
     * 11c. Update DP percentage untuk order DP (admin)
     */
    public function updateDpPercentage(Request $request, $id)
    {
        $order = Order::findOrFail($id);

        // Validasi: hanya untuk order dengan payment_method = dp
        if ($order->payment_method !== 'dp') {
            $errorMessage = 'Hanya order dengan metode pembayaran DP yang dapat diubah persentase DP-nya.';
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $errorMessage
                ], 400);
            }
            return back()->with('error', $errorMessage);
        }

        // Validasi persentase DP (min 15%, max 50%)
        try {
            $request->validate([
                'dp_percentage' => 'required|numeric|min:15|max:50',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            if (request()->ajax() || request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage(),
                    'errors' => $e->errors()
                ], 422);
            }
            throw $e;
        }

        $oldDpPercentage = $order->dp_percentage ?? 30.00;
        $newDpPercentage = $request->dp_percentage;

        // Update dp_percentage
        $order->update([
            'dp_percentage' => $newDpPercentage,
        ]);

        // Jika receivable sudah ada dan status belum "selesai", update paid_amount sesuai persentase baru
        if ($order->receivable && $order->status !== 'selesai') {
            $receivable = $order->receivable;
            
            // Hitung ulang DP amount berdasarkan persentase baru
            $newDpAmount = $order->total_price * ($newDpPercentage / 100);
            
            // Jika status sudah "diproses" (DP sudah diverifikasi), update paid_amount
            // Jika status masih "menunggu" atau "menunggu_verifikasi", biarkan paid_amount = 0
            if ($order->status === 'diproses' && $receivable->paid_amount > 0) {
                // Update paid_amount sesuai persentase baru
                $receivable->update([
                    'paid_amount' => $newDpAmount,
                ]);
                
                // Trigger update status receivable
                $receivable->updatePaymentStatus();
            }
        }

        $message = "Persentase DP berhasil diubah dari {$oldDpPercentage}% menjadi {$newDpPercentage}%.";
        
        // Jika request AJAX, return JSON
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
                'dp_percentage' => $newDpPercentage
            ]);
        }
        
        return back()->with('success', $message);
    }

    /**
     * 11b. Update production status order (admin)
     */
    public function updateProductionStatus(Request $request, $id)
    {
        $order = Order::findOrFail($id);

        // Validation
        $request->validate([
            'production_status' => 'required|in:menunggu,dalam_proses,selesai,dibatalkan'
        ]);

        $oldStatus = $order->production_status;
        $order->update(['production_status' => $request->production_status]);

        // Jika update production status, sync ke Production model
        if ($order->production) {
            $order->production->update([
                'status' => $request->production_status
            ]);

            // Jika production selesai, update completed_at
            if ($request->production_status === 'selesai') {
                $order->production->update(['completed_at' => now()]);
            }
        }

        return back()->with('success', "Status produksi pesanan #{$order->id} berhasil diperbarui!");
    }

    /**
     * Mark order as read (admin)
     */
    public function markAsRead($id)
    {
        try {
            $order = Order::findOrFail($id);
            $order->update(['is_read' => true]);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Pesanan berhasil ditandai sebagai dibaca'
                ]);
            }

            return back()->with('success', "Pesanan #{$order->id} berhasil ditandai sebagai dibaca!");
        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal menandai pesanan sebagai dibaca: ' . $e->getMessage()
                ], 500);
            }

            return back()->with('error', 'Gagal menandai pesanan sebagai dibaca: ' . $e->getMessage());
        }
    }

    /**
     * Backfill user_id untuk orders berdasarkan kecocokan email.
     */
    public function backfillUserIds(Request $request)
    {
        $updated = 0;
        $skipped = 0;
        DB::beginTransaction();
        try {
            Order::whereNull('user_id')
                ->whereNotNull('customer_email')
                ->orderBy('id')
                ->chunk(500, function ($orders) use (&$updated, &$skipped) {
                    foreach ($orders as $order) {
                        $user = User::where('email', $order->customer_email)->first();
                        if ($user) {
                            $order->updateQuietly(['user_id' => $user->id]);
                            $updated++;
                        } else {
                            $skipped++;
                        }
                    }
                });

            DB::commit();
            return back()->with('success', "Backfill selesai: {$updated} order terhubung ke user, {$skipped} dilewati.");
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->with('error', 'Gagal backfill: ' . $e->getMessage());
        }
    }

    /**
     * 12. User orders list
     */
    public function myOrders()
    {
        $user = Auth::user();
        
        // Ambil order berdasarkan user_id ATAU email jika user_id null
        // Gunakan case-insensitive untuk email comparison (penting untuk production)
        // FILTER: Jangan tampilkan order Midtrans yang pending terlalu lama (kemungkinan dibatalkan)
        $orders = Order::where(function($query) use ($user) {
                $query->where('user_id', (int)$user->id)
                      ->orWhere(function($q) use ($user) {
                          $q->whereNull('user_id')
                            ->whereRaw('LOWER(TRIM(customer_email)) = ?', [strtolower(trim($user->email))]);
                      });
            })
            ->where(function($query) {
                // Filter order Midtrans yang pending terlalu lama (kemungkinan dibatalkan)
                $query->where(function($q) {
                    // Bukan Midtrans
                    $q->where('payment_method', '!=', 'midtrans');
                })->orWhere(function($q) {
                    // Midtrans tapi sudah settlement (berhasil)
                    $q->where('payment_method', 'midtrans')
                      ->where('midtrans_status', 'settlement');
                })->orWhere(function($q) {
                    // Midtrans pending tapi masih baru (< 1 jam) - mungkin masih proses
                    $q->where('payment_method', 'midtrans')
                      ->where(function($subQ) {
                          $subQ->whereNull('midtrans_status')
                               ->orWhere('midtrans_status', 'pending');
                      })
                      ->where('created_at', '>=', now()->subHour());
                });
            })
            ->with(['product'])
            ->latest('created_at')
            ->paginate(10);

        return view('user.orders.index', compact('orders'));
    }

    /**
     * 10. User show order
     */
    public function myOrderShow(Order $order)
    {
        // Route model binding sudah handle authorization di Order model
        // Tapi tetap tambahkan double check untuk keamanan ekstra
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu.');
        }

        // Double check authorization (case-insensitive untuk production)
        $user = Auth::user();
        $hasAccess = false;
        
        if ($order->user_id !== null) {
            $hasAccess = ((int)$order->user_id === (int)$user->id);
        } else {
            // Case-insensitive email comparison untuk production
            $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
        }
        
        if (!$hasAccess) {
            return redirect()->route('my.orders.index')->with('error', 'Akses ditolak. Anda tidak memiliki akses ke order ini.');
        }

        // Refresh order untuk memastikan data terbaru (termasuk proof_file)
        $order->refresh();

        // Load production dengan relasi lengkap untuk tracking
        $order->load([
            'product',
            'production' => function($query) {
                $query->with([
                    'productionMaterials.material',
                    'productionSpareparts.sparepart',
                    'teknisi',
                    'supervisor'
                ]);
            }
        ]);
        
        return view('user.orders.show', compact('order'));
    }

    /**
     * Cancel Midtrans order (dipanggil saat user tutup popup)
     */
    public function cancelMidtransOrder(Request $request, $id)
    {
        try {
            if (!Auth::check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Silakan login terlebih dahulu.',
                ], 401);
            }

            $order = Order::find($id);
            if (!$order) {
                return response()->json([
                    'success' => false,
                    'message' => 'Order tidak ditemukan.',
                ], 404);
            }

            // Authorization check
            $user = Auth::user();
            $hasAccess = false;
            
            if ($order->user_id !== null) {
                $hasAccess = ((int)$order->user_id === (int)$user->id);
            } else {
                $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
            }
            
            if (!$hasAccess) {
                return response()->json([
                    'success' => false,
                    'message' => 'Akses ditolak.',
                ], 403);
            }

            // Hanya bisa cancel order Midtrans yang masih pending
            if ($order->payment_method !== 'midtrans') {
                return response()->json([
                    'success' => false,
                    'message' => 'Hanya bisa membatalkan order Midtrans.',
                ], 400);
            }

            // Hanya bisa cancel jika belum settlement
            if ($order->midtrans_status === 'settlement' || $order->status === 'selesai') {
                return response()->json([
                    'success' => false,
                    'message' => 'Order yang sudah berhasil tidak bisa dibatalkan.',
                ], 400);
            }

            DB::beginTransaction();
            try {
                // Hapus production jika ada
                if ($order->production) {
                    $order->production->delete();
                }
                
                // Hapus receivable jika ada
                if ($order->receivable) {
                    $order->receivable->delete();
                }
                
                // Hapus order
                $order->delete();
                
                DB::commit();
                
                Log::info('Midtrans order cancelled by user (onClose)', [
                    'order_id' => $id,
                    'user_id' => $user->id,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Order berhasil dibatalkan.',
                ]);
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error cancelling Midtrans order: ' . $e->getMessage(), [
                    'order_id' => $id,
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal membatalkan order: ' . $e->getMessage(),
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Fatal error cancelling Midtrans order: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan server.',
            ], 500);
        }
    }

    /**
     * Retry Midtrans payment (generate new snap token)
     */
    public function retryMidtransPayment(Request $request, $id)
    {
        try {
            if (!Auth::check()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Silakan login terlebih dahulu.',
                ], 401);
            }

            $order = Order::find($id);
            if (!$order) {
                return response()->json([
                    'success' => false,
                    'message' => 'Order tidak ditemukan.',
                ], 404);
            }

            // Authorization check
            $user = Auth::user();
            $hasAccess = false;
            
            if ($order->user_id !== null) {
                $hasAccess = ((int)$order->user_id === (int)$user->id);
            } else {
                $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
            }
            
            if (!$hasAccess) {
                return response()->json([
                    'success' => false,
                    'message' => 'Akses ditolak.',
                ], 403);
            }

            // Hanya bisa retry order Midtrans yang belum selesai
            if ($order->payment_method !== 'midtrans') {
                return response()->json([
                    'success' => false,
                    'message' => 'Hanya bisa mengulang pembayaran untuk order Midtrans.',
                ], 400);
            }

            // Tidak bisa retry jika sudah settlement
            if ($order->midtrans_status === 'settlement' || $order->status === 'selesai') {
                return response()->json([
                    'success' => false,
                    'message' => 'Pembayaran sudah selesai, tidak perlu diulang.',
                ], 400);
            }

            // Reset status Midtrans sebelum retry untuk menghindari konflik
            DB::beginTransaction();
            try {
                // Reset status Midtrans sebelum retry
                $order->update([
                    'midtrans_status' => null,
                    'midtrans_transaction_id' => null,
                    'midtrans_payment_type' => null,
                    'midtrans_va_number' => null,
                    'status' => 'menunggu', // Reset ke menunggu
                ]);

                $product = $order->product;
                if (!$product) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Produk tidak ditemukan.',
                    ], 404);
                }

                // Gunakan isRetry = true untuk membuat order_id unik
                $snapResponse = $this->generateSnapToken($order, $product, true);
                
                DB::commit();
                
                return response()->json([
                    'success' => true,
                    'snap_token' => $snapResponse['snap_token'],
                    'order_id' => $order->id,
                    'midtrans_order_id' => $snapResponse['midtrans_order_id'] ?? null,
                    'message' => 'Token pembayaran berhasil dibuat. Silakan lakukan pembayaran.',
                ]);
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error retrying Midtrans payment: ' . $e->getMessage(), [
                    'order_id' => $id,
                    'trace' => $e->getTraceAsString(),
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal membuat token pembayaran: ' . $e->getMessage(),
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Fatal error retrying Midtrans payment: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan server.',
            ], 500);
        }
    }

    /**
     * Change payment method for existing order
     */
    public function changePaymentMethod(Request $request, $id)
    {
        try {
            if (!Auth::check()) {
                return redirect()->route('login')->with('error', 'Silakan login terlebih dahulu.');
            }

            $order = Order::find($id);
            if (!$order) {
                return back()->with('error', 'Order tidak ditemukan.');
            }

            // Authorization check
            $user = Auth::user();
            $hasAccess = false;
            
            if ($order->user_id !== null) {
                $hasAccess = ((int)$order->user_id === (int)$user->id);
            } else {
                $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
            }
            
            if (!$hasAccess) {
                return back()->with('error', 'Akses ditolak.');
            }

            // Validasi: hanya bisa ganti jika pembayaran belum selesai
            if ($order->status === 'selesai' || $order->midtrans_status === 'settlement') {
                return back()->with('error', 'Pembayaran sudah selesai, tidak bisa mengganti metode pembayaran.');
            }

            $request->validate([
                'new_payment_method' => 'required|string|in:transfer,midtrans,dp',
            ]);

            $newPaymentMethod = $request->new_payment_method;

            // Jika ganti ke Midtrans, generate snap token
            if ($newPaymentMethod === 'midtrans') {
                DB::beginTransaction();
                try {
                    $order->update([
                        'payment_method' => 'midtrans',
                        'status' => 'menunggu',
                        'midtrans_status' => null,
                        'midtrans_transaction_id' => null,
                    ]);

                    $product = $order->product;
                    if (!$product) {
                        throw new \Exception('Produk tidak ditemukan.');
                    }

                    // Jika ganti ke Midtrans, gunakan isRetry = true karena ini retry dari metode lain
                    $snapResponse = $this->generateSnapToken($order, $product, true);
                    DB::commit();

                    return response()->json([
                        'success' => true,
                        'snap_token' => $snapResponse['snap_token'],
                        'order_id' => $order->id,
                        'midtrans_order_id' => $snapResponse['midtrans_order_id'] ?? null,
                        'message' => 'Metode pembayaran berhasil diubah ke Midtrans.',
                    ]);
                } catch (\Exception $e) {
                    DB::rollBack();
                    Log::error('Error changing payment method to Midtrans: ' . $e->getMessage());
                    return response()->json([
                        'success' => false,
                        'message' => 'Gagal mengubah metode pembayaran: ' . $e->getMessage(),
                    ], 500);
                }
            } else {
                // Ganti ke metode lain (transfer atau dp)
                $order->update([
                    'payment_method' => $newPaymentMethod,
                    'status' => $newPaymentMethod === 'dp' ? 'menunggu' : 'menunggu',
                    'midtrans_status' => null,
                    'midtrans_transaction_id' => null,
                ]);

                if ($newPaymentMethod === 'transfer') {
                    return redirect()->route('orders.confirm.form', $order)
                        ->with('success', 'Metode pembayaran berhasil diubah ke Transfer Bank. Silakan upload bukti transfer.');
                } else {
                    return redirect()->route('my.orders.show', $order)
                        ->with('success', 'Metode pembayaran berhasil diubah ke DP (Down Payment).');
                }
            }
        } catch (\Exception $e) {
            Log::error('Fatal error changing payment method: ' . $e->getMessage());
            return back()->with('error', 'Terjadi kesalahan: ' . $e->getMessage());
        }
    }

    /**
     * Show payment proof file with authorization check
     */
    public function showProof($id)
    {
        // Cek apakah user login
        if (!Auth::check()) {
            abort(403, 'Unauthorized');
        }

        // Cari order berdasarkan ID (tanpa route model binding untuk menghindari issue di production)
        $order = Order::find($id);
        
        if (!$order) {
            Log::warning('Order not found for proof access', [
                'order_id' => $id,
                'user_id' => Auth::id(),
            ]);
            abort(404, 'Order not found.');
        }

        // Double check authorization (case-insensitive untuk production)
        $user = Auth::user();
        $hasAccess = false;
        
        // Admin bisa akses semua order proof
        if ($user->user_type === 'admin') {
            $hasAccess = true;
        } else {
            // User biasa hanya bisa akses order mereka sendiri
            if ($order->user_id !== null) {
                $hasAccess = ((int)$order->user_id === (int)$user->id);
            } else {
                // Case-insensitive email comparison untuk production
                $hasAccess = (strtolower(trim($order->customer_email)) === strtolower(trim($user->email)));
            }
        }
        
        if (!$hasAccess) {
            Log::warning('Unauthorized access attempt to order proof', [
                'order_id' => $order->id,
                'order_user_id' => $order->user_id,
                'order_email' => $order->customer_email,
                'current_user_id' => $user->id,
                'current_user_email' => $user->email,
                'current_user_type' => $user->user_type,
            ]);
            abort(403, 'You do not have access to this order.');
        }

        // Cek apakah proof_file ada
        if (!$order->proof_file) {
            abort(404, 'Payment proof not found.');
        }

        // Cek apakah file benar-benar ada di storage
        if (!Storage::disk('uploads')->exists($order->proof_file)) {
            Log::warning('Payment proof file not found in storage', [
                'order_id' => $order->id,
                'proof_file' => $order->proof_file,
                'storage_path' => public_path('uploads'),
                'full_path' => public_path('uploads/' . $order->proof_file),
            ]);
            abort(404, 'Payment proof file not found in storage.');
        }

        try {
            // Get file path - gunakan path() untuk mendapatkan full path
            $filePath = Storage::disk('uploads')->path($order->proof_file);
            
            // Log untuk debugging di production
            Log::info('Serving payment proof', [
                'order_id' => $order->id,
                'proof_file' => $order->proof_file,
                'file_path' => $filePath,
                'file_exists' => file_exists($filePath),
            ]);
            
            // Cek apakah file benar-benar ada
            if (!file_exists($filePath)) {
                Log::error('Payment proof file not found on disk', [
                    'order_id' => $order->id,
                    'proof_file' => $order->proof_file,
                    'expected_path' => $filePath,
                    'storage_path' => public_path('uploads'),
                ]);
                abort(404, 'Payment proof file not found on disk.');
            }
            
            // Get file mime type
            $mimeType = Storage::disk('uploads')->mimeType($order->proof_file);
            if (!$mimeType) {
                // Fallback mime type berdasarkan extension
                $extension = pathinfo($order->proof_file, PATHINFO_EXTENSION);
                $mimeType = match(strtolower($extension)) {
                    'jpg', 'jpeg' => 'image/jpeg',
                    'png' => 'image/png',
                    'pdf' => 'application/pdf',
                    default => 'application/octet-stream',
                };
            }
            
            // Return file response dengan headers yang benar
            return response()->file($filePath, [
                'Content-Type' => $mimeType,
                'Content-Disposition' => 'inline; filename="' . basename($order->proof_file) . '"',
                'Cache-Control' => 'private, max-age=3600',
            ]);
        } catch (\Exception $e) {
            Log::error('Error serving payment proof: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'proof_file' => $order->proof_file,
                'file_path' => $filePath ?? 'N/A',
                'trace' => $e->getTraceAsString(),
            ]);
            abort(500, 'Error loading payment proof file: ' . $e->getMessage());
        }
    }
}
