<?php

namespace App\Http\Controllers\Supervisor;

use App\Http\Controllers\Controller;
use App\Models\Production;
use App\Models\Product;
use App\Models\Sale;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Material;
use App\Models\Sparepart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    public function productions(Request $request)
    {
        $query = Production::with([
            'order',
            'product',
            'teknisi',
            'productionMaterials.material',
            'productionSpareparts.sparepart',
        ])->orderBy('created_at', 'desc');

        if ($request->filled('start_date')) {
            $query->where('created_at', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('created_at', '<=', $request->end_date . ' 23:59:59');
        }
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        $productions = $query->paginate(20)->withQueryString();

        $totalProductions = (clone $query)->count();
        $totalMaterialCost = (clone $query)->sum('total_material_cost') ?? 0;
        $totalSparepartCost = (clone $query)->sum('total_sparepart_cost') ?? 0;
        $totalLaborCost = (clone $query)->sum('labor_cost') ?? 0;
        $totalProductionCost = $totalMaterialCost + $totalSparepartCost + $totalLaborCost;

        $productList = Product::where('is_active', true)->get();

        $chartData = Production::select(
                DB::raw('MONTH(created_at) as month'),
                DB::raw('YEAR(created_at) as year'),
                DB::raw('COUNT(*) as count'),
                DB::raw('SUM(total_production_cost) as total_cost'),
                DB::raw('SUM(labor_cost) as total_labor')
            )
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('supervisor.reports.productions', compact(
            'productions',
            'totalProductions',
            'totalMaterialCost',
            'totalSparepartCost',
            'totalLaborCost',
            'totalProductionCost',
            'productList',
            'chartData'
        ));
    }

    /**
     * Laporan Penjualan
     */
    public function sales(Request $request)
    {
        // Query tanpa cache untuk memastikan data selalu fresh/real-time
        // Gunakan created_at sebagai prioritas pertama untuk memastikan data terbaru selalu muncul di atas
        $query = Sale::with(['product', 'user', 'order'])
            ->orderBy('created_at', 'desc') // Prioritas pertama: data terbaru berdasarkan waktu dibuat
            ->orderBy('sale_date', 'desc'); // Fallback: jika created_at sama, urutkan berdasarkan sale_date

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('sale_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('sale_date', '<=', $request->end_date . ' 23:59:59');
        }

        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Paginate dengan withQueryString untuk mempertahankan filter saat navigasi
        $sales = $query->paginate(20)->withQueryString();
        
        // Statistik - hitung ulang dari query yang sama untuk memastikan konsistensi dan real-time
        // Clone query untuk menghindari modifikasi query utama
        $statsQuery = clone $query;
        $totalSales = $statsQuery->sum('total_price') ?? 0;
        
        $statsQuery = clone $query;
        $totalQuantity = $statsQuery->sum('quantity') ?? 0;
        
        $statsQuery = clone $query;
        $totalTransactions = $statsQuery->count();

        // Chart data untuk 12 bulan terakhir
        $chartData = Sale::select(
                DB::raw('MONTH(sale_date) as month'),
                DB::raw('YEAR(sale_date) as year'),
                DB::raw('SUM(total_price) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('sale_date', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        // Return view dengan header no-cache untuk memastikan data selalu fresh
        return response()
            ->view('supervisor.reports.sales', compact(
                'sales', 
                'totalSales', 
                'totalQuantity', 
                'totalTransactions',
                'chartData'
            ))
            ->header('Cache-Control', 'no-cache, no-store, must-revalidate')
            ->header('Pragma', 'no-cache')
            ->header('Expires', '0');
    }

    /**
     * Laporan Pembelian
     */
    public function purchases(Request $request)
    {
        $query = Purchase::with(['items'])
            ->orderBy('purchase_date', 'desc');

        // Filter berdasarkan tanggal
        if ($request->filled('start_date')) {
            $query->where('purchase_date', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $query->where('purchase_date', '<=', $request->end_date);
        }

        // Filter berdasarkan supplier
        if ($request->filled('supplier')) {
            $query->where('supplier_name', 'like', '%' . $request->supplier . '%');
        }

        // Filter berdasarkan tipe pembelian
        if ($request->filled('purchase_type')) {
            $query->where('purchase_type', $request->purchase_type);
        }

        $purchases = $query->paginate(20);

        // Statistik
        $totalAmount = (clone $query)->sum('total_amount');
        $totalTransactions = (clone $query)->count();

        // Detail items untuk statistik
        $purchaseItems = PurchaseItem::with(['purchase'])
            ->whereHas('purchase', function($q) use ($request) {
                if ($request->filled('start_date')) {
                    $q->where('purchase_date', '>=', $request->start_date);
                }
                if ($request->filled('end_date')) {
                    $q->where('purchase_date', '<=', $request->end_date);
                }
            })
            ->get();

        $totalItems = $purchaseItems->sum('quantity');
        $totalItemValue = $purchaseItems->sum('total_price');

        // Chart data untuk 12 bulan terakhir
        $chartData = Purchase::select(
                DB::raw('MONTH(purchase_date) as month'),
                DB::raw('YEAR(purchase_date) as year'),
                DB::raw('SUM(total_amount) as total'),
                DB::raw('COUNT(*) as count')
            )
            ->where('purchase_date', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'asc')
            ->orderBy('month', 'asc')
            ->get();

        return view('supervisor.reports.purchases', compact(
            'purchases',
            'totalAmount',
            'totalTransactions',
            'totalItems',
            'totalItemValue',
            'chartData'
        ));
    }

    /**
     * Laporan Barang Limit (Stock yang hampir habis)
     */
    public function stockLimit(Request $request)
    {
        // Ambil materials dengan stock rendah
        $materialsQuery = Material::where('is_active', true)
            ->where('stock', '<=', 10); // Threshold bisa disesuaikan

        if ($request->filled('category')) {
            $materialsQuery->where('category', $request->category);
        }

        $materials = $materialsQuery->orderBy('stock', 'asc')->get();

        // Ambil spareparts dengan stock rendah
        $sparepartsQuery = Sparepart::where('is_active', true)
            ->where('stock', '<=', 5); // Threshold bisa disesuaikan

        if ($request->filled('brand')) {
            $sparepartsQuery->where('brand', $request->brand);
        }

        $spareparts = $sparepartsQuery->orderBy('stock', 'asc')->get();

        // Statistik
        $totalMaterialsLow = $materials->count();
        $totalSparepartsLow = $spareparts->count();
        $totalLowStock = $totalMaterialsLow + $totalSparepartsLow;

        // Hitung nilai total stock rendah
        $totalValueMaterials = $materials->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueSpareparts = $spareparts->sum(function($item) {
            return $item->stock * $item->price;
        });

        $totalValueLowStock = $totalValueMaterials + $totalValueSpareparts;

        return view('supervisor.reports.stock-limit', compact(
            'materials',
            'spareparts',
            'totalMaterialsLow',
            'totalSparepartsLow',
            'totalLowStock',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueLowStock'
        ));
    }

    /**
     * Laporan Barang Rusak
     */
    public function damagedItems(Request $request)
    {
        // Menggunakan scope hasDamage jika ada, atau filter manual
        $materialsQuery = Material::where(function($q) {
            $q->where('damaged_stock', '>', 0)
              ->orWhere('condition', 'damaged');
        });

        $sparepartsQuery = Sparepart::where(function($q) {
            $q->where('damaged_stock', '>', 0)
              ->orWhere('condition', 'damaged');
        });

        $productsQuery = Product::where('condition', 'damaged');

        // Filter berdasarkan kategori (untuk materials)
        if ($request->filled('category')) {
            $materialsQuery->where('category', $request->category);
        }

        // Filter berdasarkan brand (untuk spareparts)
        if ($request->filled('brand')) {
            $sparepartsQuery->where('brand', $request->brand);
        }

        $materials = $materialsQuery->orderBy('updated_at', 'desc')->get();
        $spareparts = $sparepartsQuery->orderBy('updated_at', 'desc')->get();
        $products = $productsQuery->orderBy('updated_at', 'desc')->get();

        // Statistik
        $totalDamagedMaterials = $materials->count();
        $totalDamagedSpareparts = $spareparts->count();
        $totalDamagedProducts = $products->count();
        $totalDamagedItems = $totalDamagedMaterials + $totalDamagedSpareparts + $totalDamagedProducts;

        // Hitung nilai total barang rusak
        $totalValueMaterials = $materials->sum(function($item) {
            return ($item->damaged_stock ?? 0) * $item->price;
        });

        $totalValueSpareparts = $spareparts->sum(function($item) {
            return ($item->damaged_stock ?? 0) * $item->price;
        });

        $totalValueProducts = $products->sum(function($item) {
            return $item->product_prices ?? 0;
        });

        $totalValueDamaged = $totalValueMaterials + $totalValueSpareparts + $totalValueProducts;

        return view('supervisor.reports.damaged-items', compact(
            'materials',
            'spareparts',
            'products',
            'totalDamagedMaterials',
            'totalDamagedSpareparts',
            'totalDamagedProducts',
            'totalDamagedItems',
            'totalValueMaterials',
            'totalValueSpareparts',
            'totalValueProducts',
            'totalValueDamaged'
        ));
    }
}


