<?php

namespace App\Http\Controllers\Teknisi;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Production;
use App\Models\Material;
use App\Models\Sparepart;
use App\Models\ProductionMaterial;
use App\Models\ProductionSparepart;
use App\Models\ProductionItemRequest;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ProductionController extends Controller
{
    public function index(Request $request)
    {
        $query = Production::with([
            'order', 
            'product', 
            'teknisi'
        ])
            ->where('teknisi_id', Auth::id())
            ->whereNotNull('teknisi_id')
            ->where('status', '!=', 'selesai');

        // Search functionality
        if ($request->filled('search') && trim($request->search) !== '') {
            $searchTerm = trim($request->search);
            $query->where(function($q) use ($searchTerm) {
                $q->whereHas('product', function($q2) use ($searchTerm) {
                    $q2->where('product_title', 'like', "%{$searchTerm}%");
                })
                ->orWhereHas('order', function($q2) use ($searchTerm) {
                    $q2->where('customer_name', 'like', "%{$searchTerm}%")
                       ->orWhere('customer_email', 'like', "%{$searchTerm}%");
                    // Jika search adalah numerik, cari juga ID order
                    if (is_numeric($searchTerm)) {
                        $q2->orWhere('id', $searchTerm);
                    }
                });
                // Jika search adalah numerik, cari juga ID production
                if (is_numeric($searchTerm)) {
                    $q->orWhere('id', $searchTerm);
                } else {
                    $q->orWhere('id', 'like', "%{$searchTerm}%");
                }
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by planning status
        if ($request->filled('planning_status')) {
            $query->where('planning_status', $request->planning_status);
        }

        $productions = $query->latest('assigned_at')->paginate(15)->withQueryString();
        
        // Ambil ID produksi yang memiliki notifikasi belum dibaca
        $teknisiId = Auth::id();
        $productionsWithNotifications = Notification::where('user_id', $teknisiId)
            ->where('type', 'production')
            ->where('is_read', false)
            ->whereNotNull('related_production_id')
            ->pluck('related_production_id')
            ->unique()
            ->toArray();
        
        // Tambahkan flag has_unread_notification ke setiap produksi
        $productions->getCollection()->transform(function($production) use ($productionsWithNotifications) {
            $production->has_unread_notification = in_array($production->id, $productionsWithNotifications);
            return $production;
        });
        
        return view('teknisi.productions.index', compact('productions'));
    }

    public function history()
    {
        $productions = Production::with(['order', 'product', 'teknisi'])
            ->where('teknisi_id', Auth::id())
            ->whereNotNull('teknisi_id')
            ->where('status', 'selesai')
            ->latest('updated_at')
            ->paginate(20);
        
        return view('teknisi.productions.history', compact('productions'));
    }

    public function showHistory(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan production sudah selesai (history)
        if ($production->status !== 'selesai') {
            return redirect()->route('teknisi.productions.show', $production)
                ->with('error', 'Produksi ini belum selesai. Gunakan halaman detail produksi aktif.');
        }

        // Load relasi yang diperlukan dengan eager loading
        $production->load([
            'order', 
            'product.specifications',
            'product.images',
            'product.category',
            'teknisi',
            'supervisor',
            'completionApprover',
            'materialsReceiver',
            'sparepartsReceiver',
            'productionMaterials.material', 
            'productionSpareparts.sparepart'
        ]);
        
        return view('teknisi.productions.history-show', compact('production'));
    }

    public function show(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        // Cek apakah teknisi_id ada dan sesuai dengan user yang login
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Load relasi yang diperlukan dengan eager loading
        $production->load([
            'order', 
            'product.specifications',
            'product.images',
            'product.category',
            'teknisi',
            'supervisor',
            'completionApprover',
            'materialsReceiver',
            'sparepartsReceiver',
            'productionMaterials.material',
            'productionMaterials.creator',
            'productionMaterials.reviser',
            'productionSpareparts.sparepart',
            'productionSpareparts.creator',
            'productionSpareparts.reviser',
            'itemRequests.material',
            'itemRequests.sparepart',
            'itemRequests.requester',
            'itemRequests.processor'
        ]);
        
        // Ambil semua bahan dan sparepart untuk dropdown
        $materials = Material::where('is_active', true)->get();
        $spareparts = Sparepart::where('is_active', true)->get();
        
        // Gabungkan materials dan spareparts untuk tabel unified
        $allItems = $this->getCombinedProductionItems($production);
        
        // Group item requests by proposal number
        $itemRequestsByProposal = $production->itemRequests->groupBy('proposal_number');
        
        return view('teknisi.productions.show', compact('production', 'materials', 'spareparts', 'allItems', 'itemRequestsByProposal'));
    }

    /**
     * Gabungkan materials dan spareparts menjadi satu collection dengan tipe sebagai pembeda
     */
    private function getCombinedProductionItems(Production $production)
    {
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        
        $items = collect();
        
        // Add materials
        foreach ($production->productionMaterials as $pm) {
            $totalNeeded = $pm->quantity * $orderQty;
            $currentStock = $pm->material->stock ?? 0;
            $stockStatus = $currentStock >= $totalNeeded ? 'available' : 'insufficient';
            
            $items->push([
                'id' => $pm->id,
                'type' => 'material',
                'type_label' => 'Bahan',
                'item_id' => $pm->material_id,
                'name' => $pm->material->name ?? 'N/A',
                'quantity' => $pm->quantity,
                'unit' => $pm->unit,
                'unit_cost' => $pm->unit_cost,
                'total_cost' => $pm->total_cost,
                'current_stock' => $currentStock,
                'total_needed' => $totalNeeded,
                'stock_status' => $stockStatus,
                'stock_status_label' => $stockStatus === 'available' ? 'Tersedia' : 'Perlu Pembelian',
                'is_received' => $pm->is_received,
                'received_at' => $pm->received_at,
                'is_additional' => $pm->is_additional,
                'model' => $pm,
            ]);
        }
        
        // Add spareparts
        foreach ($production->productionSpareparts as $ps) {
            $totalNeeded = $ps->quantity * $orderQty;
            $currentStock = $ps->sparepart->stock ?? 0;
            $stockStatus = $currentStock >= $totalNeeded ? 'available' : 'insufficient';
            
            $items->push([
                'id' => $ps->id,
                'type' => 'sparepart',
                'type_label' => 'Sparepart',
                'item_id' => $ps->sparepart_id,
                'name' => $ps->sparepart->name ?? 'N/A',
                'quantity' => $ps->quantity,
                'unit' => $ps->unit,
                'unit_cost' => $ps->unit_cost,
                'total_cost' => $ps->total_cost,
                'current_stock' => $currentStock,
                'total_needed' => $totalNeeded,
                'stock_status' => $stockStatus,
                'stock_status_label' => $stockStatus === 'available' ? 'Tersedia' : 'Perlu Pembelian',
                'is_received' => $ps->is_received,
                'received_at' => $ps->received_at,
                'is_additional' => $ps->is_additional,
                'model' => $ps,
            ]);
        }
        
        return $items;
    }

    public function addMaterials(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa menambah bahan. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa menambah bahan. Status planning sudah berubah.']);
        }

        $request->validate([
            'material_id' => 'required|exists:materials,id',
            'quantity' => 'required|numeric|min:0.01',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
        ]);

        // Cek apakah material sudah ada di production ini (mencegah duplikasi)
        $existingMaterial = $production->productionMaterials()
            ->where('material_id', $request->material_id)
            ->with('material')
            ->first();
        
        if ($existingMaterial) {
            $materialName = $existingMaterial->material->name ?? 'Material';
            if (request()->expectsJson()) {
                return response()->json(['error' => "Bahan '{$materialName}' sudah ada di planning. Jika jumlahnya salah, silakan edit atau hapus terlebih dahulu."], 422);
            }
            return back()->withErrors(['material_id' => "Bahan '{$materialName}' sudah ada di planning. Jika jumlahnya salah, silakan edit atau hapus terlebih dahulu."]);
        }

        $material = Material::findOrFail($request->material_id);
        
        // Hitung order quantity (dari production atau order)
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        $orderQty = max(1, (int) $orderQty);
        
        // Total cost = quantity per unit * unit cost * order quantity
        $totalCost = $request->quantity * $request->unit_cost * $orderQty;
        
        // Total quantity yang dibutuhkan untuk seluruh order
        $totalQuantityNeeded = $request->quantity * $orderQty;

        // Cek stok (hanya warning, tidak langsung kurangi stok)
        // Cek stok berdasarkan total quantity yang dibutuhkan untuk seluruh order
        if ($material->stock < $totalQuantityNeeded) {
            if (request()->expectsJson()) {
                return response()->json(['error' => "Stok bahan tidak mencukupi. Stok tersedia: {$material->stock}, butuh: {$totalQuantityNeeded} untuk {$orderQty} unit produk"], 422);
            }
            return back()->withErrors(['quantity' => "Stok bahan tidak mencukupi. Stok tersedia: {$material->stock}, butuh: {$totalQuantityNeeded} untuk {$orderQty} unit produk"]);
        }

        // Tambahkan ke production materials (PLANNING - belum kurangi stok)
        $productionMaterial = ProductionMaterial::create([
            'production_id' => $production->id,
            'material_id' => $request->material_id,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'unit_cost' => $request->unit_cost,
            'total_cost' => $totalCost,
        ]);

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Bahan berhasil ditambahkan ke planning produksi.',
                'item' => [
                    'id' => $productionMaterial->id,
                    'material_id' => $productionMaterial->material_id,
                    'name' => $material->name,
                    'quantity' => $productionMaterial->quantity,
                    'unit' => $productionMaterial->unit,
                    'unit_cost' => $productionMaterial->unit_cost,
                    'total_cost' => $productionMaterial->total_cost,
                    'stock' => $material->stock,
                ],
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Bahan berhasil ditambahkan ke planning produksi.');
    }

    public function addSpareparts(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa menambah sparepart. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa menambah sparepart. Status planning sudah berubah.']);
        }

        $request->validate([
            'sparepart_id' => 'required|exists:spareparts,id',
            'quantity' => 'required|numeric|min:0.01',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
        ]);

        // Cek apakah sparepart sudah ada di production ini (mencegah duplikasi)
        $existingSparepart = $production->productionSpareparts()
            ->where('sparepart_id', $request->sparepart_id)
            ->with('sparepart')
            ->first();
        
        if ($existingSparepart) {
            $sparepartName = $existingSparepart->sparepart->name ?? 'Sparepart';
            if (request()->expectsJson()) {
                return response()->json(['error' => "Sparepart '{$sparepartName}' sudah ada di planning. Jika jumlahnya salah, silakan edit atau hapus terlebih dahulu."], 422);
            }
            return back()->withErrors(['sparepart_id' => "Sparepart '{$sparepartName}' sudah ada di planning. Jika jumlahnya salah, silakan edit atau hapus terlebih dahulu."]);
        }

        $sparepart = Sparepart::findOrFail($request->sparepart_id);
        
        // Hitung order quantity (dari production atau order)
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        $orderQty = max(1, (int) $orderQty);
        
        // Total cost = quantity per unit * unit cost * order quantity
        $totalCost = $request->quantity * $request->unit_cost * $orderQty;
        
        // Total quantity yang dibutuhkan untuk seluruh order
        $totalQuantityNeeded = $request->quantity * $orderQty;

        // Cek stok (hanya warning, tidak langsung kurangi stok)
        // Cek stok berdasarkan total quantity yang dibutuhkan untuk seluruh order
        if ($sparepart->stock < $totalQuantityNeeded) {
            if (request()->expectsJson()) {
                return response()->json(['error' => "Stok sparepart tidak mencukupi. Stok tersedia: {$sparepart->stock}, butuh: {$totalQuantityNeeded} untuk {$orderQty} unit produk"], 422);
            }
            return back()->withErrors(['quantity' => "Stok sparepart tidak mencukupi. Stok tersedia: {$sparepart->stock}, butuh: {$totalQuantityNeeded} untuk {$orderQty} unit produk"]);
        }

        // Tambahkan ke production spareparts (PLANNING - belum kurangi stok)
        $productionSparepart = ProductionSparepart::create([
            'production_id' => $production->id,
            'sparepart_id' => $request->sparepart_id,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'unit_cost' => $request->unit_cost,
            'total_cost' => $totalCost,
        ]);

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Sparepart berhasil ditambahkan ke planning produksi.',
                'item' => [
                    'id' => $productionSparepart->id,
                    'sparepart_id' => $productionSparepart->sparepart_id,
                    'name' => $sparepart->name,
                    'quantity' => $productionSparepart->quantity,
                    'unit' => $productionSparepart->unit,
                    'unit_cost' => $productionSparepart->unit_cost,
                    'total_cost' => $productionSparepart->total_cost,
                    'stock' => $sparepart->stock,
                ],
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Sparepart berhasil ditambahkan ke planning produksi.');
    }

    /**
     * Teknisi bisa merevisi/edit bahan yang diinput admin
     */
    public function reviseMaterial(Request $request, Production $production, ProductionMaterial $productionMaterial)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa merevisi bahan. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa merevisi bahan. Status planning sudah berubah.']);
        }

        $request->validate([
            'material_id' => 'required|exists:materials,id',
            'quantity' => 'required|numeric|min:0.01',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
        ]);

        $material = Material::findOrFail($request->material_id);
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        $orderQty = max(1, (int) $orderQty);
        $totalCost = $request->quantity * $request->unit_cost * $orderQty;

        // Update bahan dengan tracking revisi
        $productionMaterial->update([
            'material_id' => $request->material_id,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'unit_cost' => $request->unit_cost,
            'total_cost' => $totalCost,
            'revised_by' => Auth::id(),
            'revised_by_teknisi_at' => now(),
        ]);

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        // Kirim notifikasi untuk revisi bahan
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyProductionRevised($production->fresh());
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send production revision notification: ' . $e->getMessage());
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Bahan berhasil direvisi.',
                'material' => [
                    'id' => $productionMaterial->id,
                    'material_id' => $productionMaterial->material_id,
                    'name' => $material->name,
                    'quantity' => $productionMaterial->quantity,
                    'unit' => $productionMaterial->unit,
                    'unit_cost' => $productionMaterial->unit_cost,
                    'total_cost' => $productionMaterial->total_cost,
                ],
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Bahan berhasil direvisi.');
    }

    public function deleteMaterial(Production $production, ProductionMaterial $productionMaterial)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Route model binding sudah memastikan productionMaterial milik production ini

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa menghapus bahan. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa menghapus bahan. Status planning sudah berubah.']);
        }

        // Simpan material_id sebelum dihapus untuk response
        $materialId = $productionMaterial->material_id;

        $productionMaterial->delete();

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Bahan berhasil dihapus.',
                'material_id' => $materialId,
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Bahan berhasil dihapus.');
    }

    /**
     * Teknisi bisa merevisi/edit sparepart yang diinput admin
     */
    public function reviseSparepart(Request $request, Production $production, ProductionSparepart $productionSparepart)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa merevisi sparepart. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa merevisi sparepart. Status planning sudah berubah.']);
        }

        $request->validate([
            'sparepart_id' => 'required|exists:spareparts,id',
            'quantity' => 'required|numeric|min:0.01',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'required|numeric|min:0',
        ]);

        $sparepart = Sparepart::findOrFail($request->sparepart_id);
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;
        $orderQty = max(1, (int) $orderQty);
        $totalCost = $request->quantity * $request->unit_cost * $orderQty;

        // Update sparepart dengan tracking revisi
        $productionSparepart->update([
            'sparepart_id' => $request->sparepart_id,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'unit_cost' => $request->unit_cost,
            'total_cost' => $totalCost,
            'revised_by' => Auth::id(),
            'revised_by_teknisi_at' => now(),
        ]);

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        // Kirim notifikasi untuk revisi sparepart
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyProductionRevised($production->fresh());
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send production revision notification: ' . $e->getMessage());
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Sparepart berhasil direvisi.',
                'sparepart' => [
                    'id' => $productionSparepart->id,
                    'sparepart_id' => $productionSparepart->sparepart_id,
                    'name' => $sparepart->name,
                    'quantity' => $productionSparepart->quantity,
                    'unit' => $productionSparepart->unit,
                    'unit_cost' => $productionSparepart->unit_cost,
                    'total_cost' => $productionSparepart->total_cost,
                ],
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Sparepart berhasil direvisi.');
    }

    public function deleteSparepart(Production $production, ProductionSparepart $productionSparepart)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Route model binding sudah memastikan productionSparepart milik production ini

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Tidak bisa menghapus sparepart. Status planning sudah berubah.'], 422);
            }
            return back()->withErrors(['planning' => 'Tidak bisa menghapus sparepart. Status planning sudah berubah.']);
        }

        // Simpan sparepart_id sebelum dihapus untuk response
        $sparepartId = $productionSparepart->sparepart_id;

        $productionSparepart->delete();

        // Update total cost
        $production->calculateCosts();

        // Reload production dengan relasi terbaru
        $production->load(['productionMaterials.material', 'productionSpareparts.sparepart']);

        if (request()->expectsJson()) {
            return response()->json([
                'success' => 'Sparepart berhasil dihapus.',
                'sparepart_id' => $sparepartId,
                'totals' => [
                    'total_material_cost' => $production->total_material_cost ?? 0,
                    'total_sparepart_cost' => $production->total_sparepart_cost ?? 0,
                    'total_production_cost' => $production->total_production_cost ?? 0,
                ]
            ]);
        }

        return back()->with('success', 'Sparepart berhasil dihapus.');
    }

    public function updateStatus(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        $request->validate([
            'status' => 'required|in:dalam_proses',
            'notes' => 'nullable|string|max:1000',
        ]);

        // Batasi agar teknisi tidak bisa langsung menyelesaikan produksi via endpoint ini
        $production->update([
            'status' => 'dalam_proses',
            'notes' => $request->notes,
            'end_date' => null,
        ]);

        $statusLabel = 'Sedang Diproduksi';
        
        return back()->with('success', "Status produksi berhasil diubah menjadi: {$statusLabel}");
    }

    public function submit(Request $request, Production $production)
    {
        // Refresh production dari database untuk memastikan data terbaru
        $production->refresh();
        
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan masih dalam tahap planning
        if ($production->planning_status !== 'planning') {
            return back()->withErrors(['planning' => 'Planning sudah disubmit atau status sudah berubah.']);
        }

        // Validasi minimal ada bahan atau sparepart
        $hasMaterials = $production->productionMaterials()->count() > 0;
        $hasSpareparts = $production->productionSpareparts()->count() > 0;

        if (!$hasMaterials && !$hasSpareparts) {
            return back()->withErrors(['submit' => 'Minimal harus ada 1 bahan atau sparepart untuk submit planning.']);
        }

        $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        // Submit planning ke supervisor
        $production->update([
            'planning_status' => 'pending_approval',
            'submitted_at' => now(),
            'notes' => $request->notes,
        ]);

        // Kirim notifikasi untuk planning disubmit
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyProductionPlanningSubmitted($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send planning submission notification: ' . $e->getMessage());
        }

        return back()->with('success', 'Planning berhasil disubmit ke supervisor untuk approval. Tunggu konfirmasi supervisor.');
    }

    /**
     * Konfirmasi bahan baku (materials) sudah diterima
     */
    public function confirmMaterialsReceived(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan materials status adalah preparing atau pending
        if (!in_array($production->materials_status, ['preparing', 'pending'])) {
            return back()->withErrors(['materials' => 'Status bahan tidak valid untuk konfirmasi.']);
        }

        // Konfirmasi materials diterima
        $production->update([
            'materials_status' => 'received',
            'materials_received_at' => now(),
            'materials_received_by' => Auth::id(),
        ]);

        // Kirim notifikasi ke admin
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyMaterialsReceived($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send materials received notification: ' . $e->getMessage());
        }

        // Jika materials dan spareparts sudah diterima, set actual start date
        if ($production->materials_status === 'received' && 
            ($production->spareparts_status === 'received' || !$production->productionSpareparts->count() > 0)) {
            $actualStartDate = now();
            $estimatedCompletionDate = null;

            if ($production->estimated_duration_days) {
                $estimatedCompletionDate = $actualStartDate->copy()->addDays($production->estimated_duration_days);
            }

            $production->update([
                'actual_start_date' => $actualStartDate,
                'estimated_completion_date' => $estimatedCompletionDate,
            ]);
        }

        return back()->with('success', 'Bahan (materials) telah dikonfirmasi diterima.');
    }

    /**
     * Konfirmasi spareparts sudah diterima
     */
    public function confirmSparepartsReceived(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan spareparts status adalah preparing atau pending
        if (!in_array($production->spareparts_status, ['preparing', 'pending'])) {
            return back()->withErrors(['spareparts' => 'Status sparepart tidak valid untuk konfirmasi.']);
        }

        // Konfirmasi spareparts diterima
        $production->update([
            'spareparts_status' => 'received',
            'spareparts_received_at' => now(),
            'spareparts_received_by' => Auth::id(),
        ]);

        // Kirim notifikasi ke admin
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifySparepartsReceived($production->fresh());
        } catch (\Exception $e) {
            \Log::error('Failed to send spareparts received notification: ' . $e->getMessage());
        }

        // Jika materials dan spareparts sudah diterima, set actual start date
        if ($production->spareparts_status === 'received' && 
            ($production->materials_status === 'received' || !$production->productionMaterials->count() > 0)) {
            $actualStartDate = now();
            $estimatedCompletionDate = null;

            if ($production->estimated_duration_days) {
                $estimatedCompletionDate = $actualStartDate->copy()->addDays($production->estimated_duration_days);
            }

            $production->update([
                'actual_start_date' => $actualStartDate,
                'estimated_completion_date' => $estimatedCompletionDate,
            ]);
        }

        return back()->with('success', 'Spareparts telah dikonfirmasi diterima.');
    }

    /**
     * Konfirmasi semua bahan dan sparepart sudah diterima sekaligus
     */
    public function confirmAllItemsReceived(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan produksi belum dimulai (status masih menunggu)
        if ($production->status !== 'menunggu') {
            return back()->withErrors(['status' => 'Produksi sudah dimulai, tidak dapat mengubah status penerimaan.']);
        }

        $updateData = [];
        $hasMaterials = $production->productionMaterials->count() > 0;
        $hasSpareparts = $production->productionSpareparts->count() > 0;

        // Konfirmasi materials jika ada (allow re-confirm jika status received tapi ada item belum diterima)
        if ($hasMaterials) {
            $allMaterialsReceived = $production->productionMaterials->every(fn($m) => $m->is_received);
            
            if (!$allMaterialsReceived || in_array($production->materials_status, ['preparing', 'pending'])) {
                $updateData['materials_status'] = 'received';
                $updateData['materials_received_at'] = now();
                $updateData['materials_received_by'] = Auth::id();
                
                // Update semua production materials
                $production->productionMaterials()->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
            }
        }

        // Konfirmasi spareparts jika ada (allow re-confirm)
        if ($hasSpareparts) {
            $allSparepartsReceived = $production->productionSpareparts->every(fn($s) => $s->is_received);
            
            if (!$allSparepartsReceived || in_array($production->spareparts_status, ['preparing', 'pending'])) {
                $updateData['spareparts_status'] = 'received';
                $updateData['spareparts_received_at'] = now();
                $updateData['spareparts_received_by'] = Auth::id();
                
                // Update semua production spareparts
                $production->productionSpareparts()->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
            }
        }

        if (!empty($updateData)) {
            $production->update($updateData);
        }

        return back()->with('success', 'Semua bahan dan sparepart telah dikonfirmasi diterima.');
    }

    public function startProduction(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah disetujui
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['approval' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan semua bahan baku (materials dan spareparts) sudah diterima
        $hasMaterials = $production->productionMaterials->count() > 0;
        $hasSpareparts = $production->productionSpareparts->count() > 0;
        
        if ($hasMaterials && $production->materials_status !== 'received') {
            return back()->withErrors(['materials' => 'Bahan (materials) belum dikonfirmasi diterima. Silakan konfirmasi terlebih dahulu.']);
        }
        
        if ($hasSpareparts && $production->spareparts_status !== 'received') {
            return back()->withErrors(['spareparts' => 'Spareparts belum dikonfirmasi diterima. Silakan konfirmasi terlebih dahulu.']);
        }

        // Gunakan actual_start_date jika sudah ada, jika tidak gunakan now()
        $startDate = $production->actual_start_date ?? now();

        // Hitung order quantity (dari production atau order)
        $orderQty = $production->quantity ?? $production->order->quantity ?? 1;

        // Mulai produksi dan kurangi stok secara atomik dengan database lock untuk mencegah race condition
        \DB::transaction(function () use ($production, $startDate, $orderQty) {
            $production->update([
                'status' => 'dalam_proses',
                'start_date' => $startDate,
                'actual_start_date' => $startDate, // Pastikan actual_start_date ter-set
            ]);

            // Sinkronkan status production ke order.production_status
            // Status production sekarang 'dalam_proses' karena teknisi sudah mulai produksi
            $production->order?->updateQuietly([
                'production_status' => 'dalam_proses'
            ]);

            // Kirim notifikasi ke admin dan supervisor
            try {
                $notificationService = new \App\Services\NotificationService();
                $notificationService->notifyProductionStarted($production->fresh());
            } catch (\Exception $e) {
                \Log::error('Failed to send production started notification: ' . $e->getMessage());
            }

            // Kurangi stok bahan dan sparepart dengan database lock untuk mencegah race condition
            // Gunakan lockForUpdate() untuk memastikan tidak ada concurrent access
            foreach ($production->productionMaterials as $pm) {
                $needed = $pm->quantity * $orderQty;
                
                // Lock material untuk update dan validasi stok
                $material = \App\Models\Material::lockForUpdate()->find($pm->material_id);
                
                if (!$material) {
                    throw new \Exception("Material dengan ID {$pm->material_id} tidak ditemukan.");
                }
                
                // Validasi stok sebelum kurangi
                if ($material->stock < $needed) {
                    throw new \Exception("Stok material '{$material->name}' tidak cukup! Stok tersedia: {$material->stock}, butuh: {$needed}");
                }
                
                // Kurangi stok
                $material->decrement('stock', $needed);
            }

            foreach ($production->productionSpareparts as $ps) {
                $needed = $ps->quantity * $orderQty;
                
                // Lock sparepart untuk update dan validasi stok
                $sparepart = \App\Models\Sparepart::lockForUpdate()->find($ps->sparepart_id);
                
                if (!$sparepart) {
                    throw new \Exception("Sparepart dengan ID {$ps->sparepart_id} tidak ditemukan.");
                }
                
                // Validasi stok sebelum kurangi
                if ($sparepart->stock < $needed) {
                    throw new \Exception("Stok sparepart '{$sparepart->name}' tidak cukup! Stok tersedia: {$sparepart->stock}, butuh: {$needed}");
                }
                
                // Kurangi stok
                $sparepart->decrement('stock', $needed);
            }
        });

        return back()->with('success', 'Produksi dimulai! Stok bahan dan sparepart telah dikurangi.');
    }

    /**
     * Revisi planning jika ditolak supervisor
     */
    public function revisePlanning(Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning status adalah rejected
        if ($production->planning_status !== 'rejected') {
            return back()->withErrors(['planning' => 'Hanya planning yang ditolak yang bisa direvisi.']);
        }

        // Reset planning status ke planning agar bisa direvisi
        // Jangan hapus supervisor_feedback agar teknisi bisa melihat feedback untuk revisi
        $production->update([
            'planning_status' => 'planning',
            // supervisor_feedback tetap disimpan agar teknisi bisa melihat feedback
        ]);

        return back()->with('success', 'Planning telah direset. Anda dapat merevisi planning sesuai feedback supervisor di bawah ini.');
    }

    /**
     * Update labor cost (upah pengerjaan)
     */
    public function updateLaborCost(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan production sudah dalam proses, selesai, atau sedang direvisi
        $canEdit = in_array($production->status, ['dalam_proses', 'selesai']) || 
                   $production->planning_status === 'planning';
        
        if (!$canEdit) {
            return back()->withErrors(['status' => 'Labor cost hanya bisa diinput saat produksi sedang berjalan, sudah selesai, atau sedang direvisi.']);
        }

        $request->validate([
            'labor_cost' => 'required|numeric|min:0',
            'notes' => 'nullable|string|max:1000',
        ]);

        $oldLaborCost = $production->labor_cost;
        $defaultServiceFee = $production->product->service_fee ?? 0;
        $newLaborCost = $request->labor_cost;

        $production->update([
            'labor_cost' => $newLaborCost,
            'notes' => $request->notes ?? $production->notes,
        ]);

        // Buat pesan sukses yang informatif
        $message = 'Labor cost berhasil diupdate.';
        if ($defaultServiceFee > 0 && $newLaborCost != $defaultServiceFee) {
            $diff = $newLaborCost - $defaultServiceFee;
            $diffFormatted = number_format(abs($diff), 0, ',', '.');
            if ($diff > 0) {
                $message .= " Nilai disesuaikan +Rp {$diffFormatted} dari default (Rp " . number_format($defaultServiceFee, 0, ',', '.') . ").";
            } else {
                $message .= " Nilai disesuaikan -Rp {$diffFormatted} dari default (Rp " . number_format($defaultServiceFee, 0, ',', '.') . ").";
            }
        }

        return back()->with('success', $message);
    }

    /**
     * Submit production completion untuk approval supervisor
     * 
     * Teknisi mengajukan penyelesaian produksi untuk disetujui supervisor.
     * Status produksi tetap 'dalam_proses' sampai supervisor menyetujui.
     * Setelah disetujui, model event akan mengubah status menjadi 'selesai'.
     */
    public function submitCompletion(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan production sedang dalam proses
        if ($production->status !== 'dalam_proses') {
            return back()->withErrors(['status' => 'Hanya production yang sedang berjalan yang bisa disubmit untuk completion.']);
        }

        $request->validate([
            'notes' => 'nullable|string|max:1000',
        ]);

        // Warning jika labor cost belum diisi (tapi tetap bisa submit)
        $hasLaborCost = $production->labor_cost && $production->labor_cost > 0;
        $warningMessage = null;
        
        if (!$hasLaborCost) {
            $warningMessage = 'Peringatan: Labor cost belum diisi. Anda tetap bisa submit, namun disarankan untuk mengisi labor cost terlebih dahulu untuk perhitungan biaya yang lebih akurat.';
        }

        // Submit completion untuk approval supervisor tanpa mengubah status menjadi selesai
        // Status akan otomatis berubah menjadi 'selesai' setelah supervisor approve via model event
        $production->update([
            'completion_status' => 'pending_approval',
            'completed_at' => now(),
            'notes' => $request->notes ?? $production->notes,
        ]);

        // Notifikasi supervisor via email
        try {
            \Illuminate\Support\Facades\Mail::send('emails.production-completion', [
                'production' => $production->load(['order', 'product', 'teknisi']),
            ], function ($mail) use ($production) {
                $mail->to(config('mail.from.address'), 'Supervisor')
                    ->subject('Produksi Selesai - Menunggu Approval - #' . $production->id)
                    ->from(config('mail.from.address'), config('mail.from.name'));
            });
        } catch (\Exception $e) {
            // Log error tapi jangan gagalkan proses
            \Illuminate\Support\Facades\Log::error('Gagal mengirim email notifikasi: ' . $e->getMessage());
        }
        
        // Kirim notifikasi untuk completion disubmit
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyProductionCompletionSubmitted($production->fresh());
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send completion submission notification: ' . $e->getMessage());
        }
        
        if ($warningMessage) {
            return back()->with('warning', $warningMessage)->with('success', 'Production completion berhasil disubmit. Menunggu approval supervisor untuk finalisasi produksi.');
        }
        
        return back()->with('success', 'Production completion berhasil disubmit. Menunggu approval supervisor untuk finalisasi produksi.');
    }

    /**
     * Toggle receive status untuk individual material (checklist)
     */
    public function toggleMaterialReceived(Request $request, Production $production, ProductionMaterial $productionMaterial)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Planning belum disetujui supervisor.'], 422);
            }
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan produksi belum dimulai
        if ($production->status !== 'menunggu') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Produksi sudah dimulai, tidak dapat mengubah status penerimaan.'], 422);
            }
            return back()->withErrors(['status' => 'Produksi sudah dimulai.']);
        }

        // Toggle status
        $newStatus = !$productionMaterial->is_received;
        $productionMaterial->update([
            'is_received' => $newStatus,
            'received_at' => $newStatus ? now() : null,
        ]);

        // Update materials_status berdasarkan kondisi semua materials
        $production->refresh();
        $allMaterialsReceived = $production->productionMaterials->every(fn($m) => $m->is_received);
        if (!$allMaterialsReceived && $production->materials_status === 'received') {
            $production->update(['materials_status' => 'preparing']);
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'is_received' => $newStatus,
                'message' => $newStatus ? 'Bahan ditandai sudah diterima.' : 'Bahan ditandai belum diterima.',
            ]);
        }

        return back()->with('success', $newStatus ? 'Bahan ditandai sudah diterima.' : 'Bahan ditandai belum diterima.');
    }

    /**
     * Toggle receive status untuk individual sparepart (checklist)
     */
    public function toggleSparepartReceived(Request $request, Production $production, ProductionSparepart $productionSparepart)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Planning belum disetujui supervisor.'], 422);
            }
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        // Pastikan produksi belum dimulai
        if ($production->status !== 'menunggu') {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Produksi sudah dimulai, tidak dapat mengubah status penerimaan.'], 422);
            }
            return back()->withErrors(['status' => 'Produksi sudah dimulai.']);
        }

        // Toggle status
        $newStatus = !$productionSparepart->is_received;
        $productionSparepart->update([
            'is_received' => $newStatus,
            'received_at' => $newStatus ? now() : null,
        ]);

        // Update spareparts_status berdasarkan kondisi semua spareparts
        $production->refresh();
        $allSparepartsReceived = $production->productionSpareparts->every(fn($s) => $s->is_received);
        if (!$allSparepartsReceived && $production->spareparts_status === 'received') {
            $production->update(['spareparts_status' => 'preparing']);
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'is_received' => $newStatus,
                'message' => $newStatus ? 'Sparepart ditandai sudah diterima.' : 'Sparepart ditandai belum diterima.',
            ]);
        }

        return back()->with('success', $newStatus ? 'Sparepart ditandai sudah diterima.' : 'Sparepart ditandai belum diterima.');
    }

    /**
     * Request item baru atau tambahan dari teknisi
     * Untuk item yang tidak tersedia atau perlu ditambah saat produksi berlangsung
     */
    public function requestItem(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Validasi: hanya bisa request saat planning approved atau dalam proses produksi
        if (!in_array($production->planning_status, ['approved']) && !in_array($production->status, ['dalam_proses'])) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Request item hanya bisa dilakukan setelah planning disetujui atau saat produksi berlangsung.'], 422);
            }
            return back()->withErrors(['status' => 'Request item hanya bisa dilakukan setelah planning disetujui atau saat produksi berlangsung.']);
        }

        $request->validate([
            'item_type' => 'required|in:material,sparepart',
            'material_id' => 'nullable|exists:materials,id',
            'sparepart_id' => 'nullable|exists:spareparts,id',
            'item_name' => 'nullable|string|max:255',
            'item_description' => 'nullable|string|max:1000',
            'quantity' => 'required|numeric|min:0.01',
            'unit' => 'required|string|max:50',
            'estimated_price' => 'nullable|numeric|min:0',
            'reason' => 'required|string|max:1000',
            'proposal_number' => 'nullable|integer|min:1|max:3',
        ]);

        // Tentukan proposal number (1, 2, atau 3)
        $proposalNumber = $request->proposal_number ?? 1;
        
        // Validasi: item harus ada (dari sistem) atau item_name harus diisi (item baru)
        if ($request->item_type === 'material' && !$request->material_id && !$request->item_name) {
            return back()->withErrors(['item' => 'Pilih bahan dari sistem atau isi nama bahan baru.']);
        }
        if ($request->item_type === 'sparepart' && !$request->sparepart_id && !$request->item_name) {
            return back()->withErrors(['item' => 'Pilih sparepart dari sistem atau isi nama sparepart baru.']);
        }

        $itemRequest = ProductionItemRequest::create([
            'production_id' => $production->id,
            'item_type' => $request->item_type,
            'material_id' => $request->item_type === 'material' ? $request->material_id : null,
            'sparepart_id' => $request->item_type === 'sparepart' ? $request->sparepart_id : null,
            'item_name' => $request->item_name,
            'item_description' => $request->item_description,
            'quantity' => $request->quantity,
            'unit' => $request->unit,
            'estimated_price' => $request->estimated_price,
            'reason' => $request->reason,
            'status' => 'pending',
            'requested_by' => Auth::id(),
            'requested_at' => now(),
            'proposal_number' => $proposalNumber,
        ]);

        // Kirim notifikasi ke admin
        try {
            $notificationService = new \App\Services\NotificationService();
            $notificationService->notifyItemRequestSubmitted($production->fresh(), $request->item_type);
        } catch (\Exception $e) {
            \Log::error('Failed to send item request submitted notification: ' . $e->getMessage());
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Permintaan item berhasil dikirim ke admin untuk diproses.',
            ]);
        }

        return back()->with('success', 'Permintaan item berhasil dikirim ke admin untuk diproses.');
    }

    /**
     * Konfirmasi penerimaan item request yang sudah dibeli
     */
    public function confirmItemRequest(Production $production, $itemRequestId)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            if (request()->expectsJson()) {
                return response()->json(['error' => 'Anda tidak memiliki akses ke produksi ini.'], 403);
            }
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        $itemRequest = ProductionItemRequest::where('id', $itemRequestId)
            ->where('production_id', $production->id)
            ->first();

        if (!$itemRequest) {
            return back()->withErrors(['item' => 'Item request tidak ditemukan.']);
        }

        // Validasi: item request harus sudah dikirim (sent) atau purchased (untuk backward compatibility)
        if (!in_array($itemRequest->status, ['sent', 'purchased'])) {
            return back()->withErrors(['status' => 'Item belum dikirim oleh admin atau sudah dikonfirmasi.']);
        }

        // Update status menjadi received
        $itemRequest->update([
            'status' => 'received',
        ]);

        // Jika item sudah ditambahkan ke production materials/spareparts, tandai sebagai received
        if ($itemRequest->item_type === 'material' && $itemRequest->material_id) {
            $productionMaterial = $production->productionMaterials()
                ->where('material_id', $itemRequest->material_id)
                ->where('is_additional', true)
                ->first();
            
            if ($productionMaterial) {
                $productionMaterial->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
            }
        } elseif ($itemRequest->item_type === 'sparepart' && $itemRequest->sparepart_id) {
            $productionSparepart = $production->productionSpareparts()
                ->where('sparepart_id', $itemRequest->sparepart_id)
                ->where('is_additional', true)
                ->first();
            
            if ($productionSparepart) {
                $productionSparepart->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
            }
        }

        if (request()->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Item berhasil dikonfirmasi diterima.',
            ]);
        }

        return back()->with('success', 'Item berhasil dikonfirmasi diterima.');
    }

    /**
     * Bulk confirm semua materials yang tersedia sebagai diterima
     */
    public function bulkConfirmMaterials(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        $request->validate([
            'material_ids' => 'required|array',
            'material_ids.*' => 'exists:production_materials,id',
        ]);

        $count = 0;
        foreach ($request->material_ids as $id) {
            $pm = ProductionMaterial::where('id', $id)
                ->where('production_id', $production->id)
                ->first();
            
            if ($pm && !$pm->is_received) {
                $pm->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
                $count++;
            }
        }

        return back()->with('success', "{$count} bahan berhasil dikonfirmasi diterima.");
    }

    /**
     * Bulk confirm semua spareparts yang tersedia sebagai diterima
     */
    public function bulkConfirmSpareparts(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan planning sudah approved
        if ($production->planning_status !== 'approved') {
            return back()->withErrors(['planning' => 'Planning belum disetujui supervisor.']);
        }

        $request->validate([
            'sparepart_ids' => 'required|array',
            'sparepart_ids.*' => 'exists:production_spareparts,id',
        ]);

        $count = 0;
        foreach ($request->sparepart_ids as $id) {
            $ps = ProductionSparepart::where('id', $id)
                ->where('production_id', $production->id)
                ->first();
            
            if ($ps && !$ps->is_received) {
                $ps->update([
                    'is_received' => true,
                    'received_at' => now(),
                ]);
                $count++;
            }
        }

        return back()->with('success', "{$count} sparepart berhasil dikonfirmasi diterima.");
    }

    /**
     * Submit hasil produksi (foto, hasil pengujian, spesifikasi)
     */
    public function submitProductionResult(Request $request, Production $production)
    {
        // Pastikan hanya teknisi yang ditugaskan yang bisa akses
        if (!$production->teknisi_id || (int)$production->teknisi_id !== (int)Auth::id()) {
            abort(403, 'Anda tidak memiliki akses ke produksi ini.');
        }

        // Pastikan production sedang dalam proses
        if ($production->status !== 'dalam_proses') {
            return back()->withErrors(['status' => 'Hanya production yang sedang berjalan yang bisa disubmit hasil.']);
        }

        $request->validate([
            'result_photos' => 'nullable|array',
            'result_photos.*' => 'image|mimes:jpeg,png,jpg,webp|max:5120',
            'test_results' => 'nullable|string|max:5000',
            'final_specifications' => 'nullable|string|max:5000',
            'completion_notes' => 'nullable|string|max:2000',
        ]);

        $resultPhotos = $production->result_photos ?? [];

        // Handle photo uploads
        if ($request->hasFile('result_photos')) {
            foreach ($request->file('result_photos') as $photo) {
                $path = $photo->store('productions/results', 'public');
                $resultPhotos[] = $path;
            }
        }

        $production->update([
            'result_photos' => $resultPhotos,
            'test_results' => $request->test_results ?? $production->test_results,
            'final_specifications' => $request->final_specifications ?? $production->final_specifications,
            'completion_notes' => $request->completion_notes ?? $production->completion_notes,
        ]);

        return back()->with('success', 'Data hasil produksi berhasil disimpan.');
    }
}

