<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ProductInquiry extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'order_id', // Relasi ke Order setelah convert
        'name',
        'email',
        'phone',
        'message',
        'is_read',
        'tracking_code',
        'admin_reply',
        'admin_replied_at',
        'reply_method',
        'user_notified',
        'status',
        'min_price',
        'max_price',
        'supervisor_note',
        'supervisor_id',
        'supervisor_approved_at',
        'estimated_days',
        'labor_cost',
        'teknisi_note',
        'teknisi_id',
        'teknisi_estimated_at',
        'final_price',
        'customer_bid_price',
        'materials_breakdown',
        'spareparts_breakdown',
        'materials_total_cost',
        'spareparts_total_cost',
        'labor_cost_total',
        'total_cost',
        'engineer_suggested_price',
        'lead_time_days',
        'supervisor_floor_price',
        'supervisor_target_price',
        'supervisor_ceiling_price',
        'validity_until',
        'final_offer_price',
        'admin_final_notes',
        'converted_to_order_at',
    ];

    protected $casts = [
        'is_read' => 'boolean',
        'user_notified' => 'boolean',
        'admin_replied_at' => 'datetime',
        'supervisor_approved_at' => 'datetime',
        'teknisi_estimated_at' => 'datetime',
        'min_price' => 'decimal:2',
        'max_price' => 'decimal:2',
        'labor_cost' => 'decimal:2',
        'final_price' => 'decimal:2',
        'customer_bid_price' => 'decimal:2',
        'materials_breakdown' => 'array',
        'spareparts_breakdown' => 'array',
        'materials_total_cost' => 'decimal:2',
        'spareparts_total_cost' => 'decimal:2',
        'labor_cost_total' => 'decimal:2',
        'total_cost' => 'decimal:2',
        'engineer_suggested_price' => 'decimal:2',
        'lead_time_days' => 'integer',
        'supervisor_floor_price' => 'decimal:2',
        'supervisor_target_price' => 'decimal:2',
        'supervisor_ceiling_price' => 'decimal:2',
        'validity_until' => 'datetime',
        'final_offer_price' => 'decimal:2',
        'converted_to_order_at' => 'datetime',
    ];

    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function supervisor()
    {
        return $this->belongsTo(User::class, 'supervisor_id');
    }

    public function teknisi()
    {
        return $this->belongsTo(User::class, 'teknisi_id');
    }

    /**
     * Relasi ke Order (setelah di-convert)
     */
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Biaya internal (materials + spareparts + labor) dengan fallback jika total_cost belum terisi
     */
    public function getInternalCostAttribute()
    {
        if (!is_null($this->total_cost)) {
            return (float) $this->total_cost;
        }
        $materials = (float) ($this->materials_total_cost ?? 0);
        $spareparts = (float) ($this->spareparts_total_cost ?? 0);
        $labor = (float) ($this->labor_cost_total ?? $this->labor_cost ?? 0);
        return $materials + $spareparts + $labor;
    }

    /**
     * Margin persen terhadap biaya internal jika final_offer_price tersedia
     */
    public function getMarginPercentAttribute()
    {
        $internal = $this->internal_cost;
        $final = (float) ($this->final_offer_price ?? $this->final_price ?? 0);
        if ($internal > 0 && $final > 0) {
            return ($final - $internal) / $internal;
        }
        return null;
    }
}
