<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class StockOpnameItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'stock_opname_id',
        'itemable_id',
        'itemable_type',
        'item_name',
        'item_code',
        'unit',
        'system_qty',
        'physical_qty',
        'difference',
        'difference_type',
        'reason',
        'notes',
    ];

    protected $casts = [
        'system_qty' => 'integer',  // atau 'float' jika butuh desimal
        'physical_qty' => 'integer', // atau 'float' jika butuh desimal
    ];

    /**
     * Polymorphic relation ke Material / Sparepart
     */
    public function itemable()
    {
        return $this->morphTo();
    }

    /**
     * Relasi ke stock opname header
     */
    public function stockOpname()
    {
        return $this->belongsTo(StockOpname::class);
    }

    /**
     * Mutator: set physical_qty otomatis menghitung selisih
     */
    public function setPhysicalQtyAttribute($value)
    {
        $this->attributes['physical_qty'] = $value;

        $systemQty = $this->system_qty ?? 0;
        $difference = $value - $systemQty;

        $this->attributes['difference'] = $difference;

        if ($difference > 0) {
            $this->attributes['difference_type'] = 'surplus';
        } elseif ($difference < 0) {
            $this->attributes['difference_type'] = 'shortage';
        } else {
            $this->attributes['difference_type'] = 'normal';
        }
    }

    /**
     * Accessor: tampilkan difference dengan format 2 desimal
     */
    public function getDifferenceFormattedAttribute()
    {
        return number_format($this->difference, 2);
    }
}
