// Real-time Notification System
class NotificationSystem {
    constructor() {
        this.pollingInterval = 30000; // 30 seconds
        this.isPolling = false;
        this.isBadgeOpen = false;
        this.isToggling = false;
        this.init();
    }

    init() {
        console.log('[NotificationSystem] Initializing...');
        // Skip initialization di halaman profile
        const currentPath = window.location.pathname;
        if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
            console.log('[NotificationSystem] Profile page detected, skipping initialization');
            return;
        }
        
        // Clear localStorage filters untuk menghindari masalah dengan notifikasi yang tidak muncul
        this.clearLocalStorageFilters();
        
        this.startPolling();
        this.bindEvents();
        console.log('[NotificationSystem] Initialized successfully');
    }

    startPolling() {
        // Skip polling di halaman profile untuk menghindari notifikasi lama muncul
        const currentPath = window.location.pathname;
        if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
            console.log('[NotificationSystem] Skipping polling on profile page');
            return;
        }
        
        if (this.isPolling) {
            console.log('[NotificationSystem] Already polling, skipping...');
            return;
        }
        
        this.isPolling = true;
        console.log('[NotificationSystem] Starting polling...');
        
        // Check immediately
        this.checkNotifications();
        
        // Then check every 30 seconds
        setInterval(() => {
            // Skip jika di halaman profile
            const currentPath = window.location.pathname;
            if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
                return;
            }
            this.checkNotifications();
        }, this.pollingInterval);
        
        // Also check when page becomes visible
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                // Skip jika di halaman profile
                const currentPath = window.location.pathname;
                if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
                    return;
                }
                console.log('[NotificationSystem] Page visible, checking notifications...');
                this.checkNotifications();
            }
        });
    }

    async checkNotifications() {
        console.log('[NotificationSystem] checkNotifications() called at:', new Date().toISOString());
        try {
            // Skip notifikasi di halaman profile untuk menghindari notifikasi lama muncul
            const currentPath = window.location.pathname;
            if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
                console.log('[NotificationSystem] Skipping notification check on profile page');
                return;
            }
            
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[NotificationSystem] CSRF token not found - cannot fetch notifications');
                return;
            }
            console.log('[NotificationSystem] CSRF token found');

            // Deteksi role berdasarkan pathname atau meta tag
            const isAdminPage = window.location.pathname.startsWith('/admin');
            const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
            const isTeknisiPage = window.location.pathname.startsWith('/teknisi');
            
            // Cek user type dari meta tag (prioritas utama)
            let userType = null;
            const userTypeMeta = document.querySelector('meta[name="user-type"]');
            if (userTypeMeta) {
                userType = userTypeMeta.getAttribute('content');
            }
            
            // Tentukan URL notifikasi berdasarkan pathname atau userType
            let notificationUrl;
            if (isAdminPage || userType === 'admin') {
                notificationUrl = '/admin/notifications';
            } else if (isSupervisorPage || userType === 'supervisor') {
                notificationUrl = '/supervisor/notifications';
            } else if (isTeknisiPage || userType === 'teknisi') {
                notificationUrl = '/teknisi/notifications';
            } else {
                // Default untuk user di halaman depan
                notificationUrl = '/notifications';
            }
            
            console.log('[NotificationSystem] Checking notifications from:', notificationUrl, 'isAdminPage:', isAdminPage, 'isSupervisorPage:', isSupervisorPage, 'isTeknisiPage:', isTeknisiPage, 'userType:', userType, 'Current path:', window.location.pathname);

            console.log('[NotificationSystem] Fetching from URL:', notificationUrl);
            const response = await fetch(notificationUrl, {
                method: 'GET',
                headers: {
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                }
            });

            console.log('[NotificationSystem] Response received:', response.status, response.statusText);
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const data = await response.json();
            
            console.log('[NotificationSystem] ===== NOTIFICATION DATA RECEIVED =====');
            console.log('[NotificationSystem] Full data:', JSON.stringify(data, null, 2));
            console.log('[NotificationSystem] Unread counts:', data.data?.unread_counts);
            console.log('[NotificationSystem] Recent items:', data.data?.recent_items);
            console.log('[NotificationSystem] Recent notifications:', data.data?.recent_items?.notifications, 'Type:', typeof data.data?.recent_items?.notifications, 'Is Array:', Array.isArray(data.data?.recent_items?.notifications), 'Length:', data.data?.recent_items?.notifications?.length);
            console.log('[NotificationSystem] ========================================');
            
            if (data.success && data.data) {
                // Skip update di halaman profile untuk menghindari notifikasi lama muncul
                const currentPath = window.location.pathname;
                if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
                    console.log('[NotificationSystem] Skipping notification update on profile page');
                    return;
                }
                
                // Gunakan data langsung dari API tanpa filter localStorage
                // API sudah melakukan filtering berdasarkan is_read = false
                // Kita hanya perlu memastikan data yang ditampilkan sesuai dengan unread_counts
                let filteredData = data.data;
                
                // Pastikan notifikasi yang ditampilkan hanya yang unread (is_read = false)
                // Ini untuk konsistensi antara badge count dan konten modal
                if (data.data.recent_items && data.data.recent_items.notifications) {
                    // Filter hanya notifikasi yang benar-benar unread
                    const unreadNotifications = data.data.recent_items.notifications.filter(notification => {
                        return !notification.is_read;
                    });
                    
                    // Update data dengan notifikasi unread saja
                    filteredData.recent_items.notifications = unreadNotifications;
                    
                    console.log('[NotificationSystem] Filtered unread notifications:', {
                        original_count: data.data.recent_items.notifications.length,
                        unread_count: unreadNotifications.length,
                        unread_counts: filteredData.unread_counts
                    });
                }
                
                // SELALU update badges (navbar dan sidebar) meskipun modal belum terbuka
                // Gunakan unread_counts dari API (lebih akurat)
                this.updatePageBadges(data.data.unread_counts);
                
                // Update content jika modal sudah terbuka
                // Gunakan filteredData yang sudah di-filter untuk unread notifications
                if (this.isBadgeOpen) {
                    this.updateNotificationContent(filteredData);
                }
            } else {
                console.error('[NotificationSystem] API returned error:', data);
            }
        } catch (error) {
            console.error('[NotificationSystem] Error checking notifications:', error);
            console.error('[NotificationSystem] Error stack:', error.stack);
            console.error('[NotificationSystem] Error details:', {
                message: error.message,
                name: error.name,
                url: window.location.href
            });
        }
    }

    updatePageBadges(unreadCounts) {
        console.log('[NotificationSystem] Updating badges:', unreadCounts);
        
            const total = unreadCounts.total || 0;
        
        // Update navbar notification badge (bisa ada beberapa di desktop dan mobile)
        const navbarBadges = document.querySelectorAll('#notification-count, #notification-count-mobile');
        navbarBadges.forEach(navbarBadge => {
            if (navbarBadge) {
            navbarBadge.textContent = total > 99 ? '99+' : total.toString();
            
            if (total > 0) {
                navbarBadge.classList.remove('hidden');
                navbarBadge.style.display = 'flex';
                    navbarBadge.style.visibility = 'visible';
            } else {
                navbarBadge.classList.add('hidden');
                navbarBadge.style.display = 'none';
                    navbarBadge.style.visibility = 'hidden';
                }
            }
        });
        
        // Fallback: jika tidak ada badge dengan id, cari dengan class badge
        if (navbarBadges.length === 0) {
            const badgeByClass = document.querySelectorAll('.badge');
            badgeByClass.forEach(badge => {
                // Cek apakah badge ini adalah notification badge (ada di dalam button notification)
                const parent = badge.closest('button');
                if (parent && (parent.id === 'notification-toggle-btn' || parent.classList.contains('notification-bell') || parent.classList.contains('notification-icon'))) {
                    badge.textContent = total > 99 ? '99+' : total.toString();
                    if (total > 0) {
                        badge.classList.remove('hidden');
                        badge.style.display = 'flex';
                        badge.style.visibility = 'visible';
        } else {
                        badge.classList.add('hidden');
                        badge.style.display = 'none';
                        badge.style.visibility = 'hidden';
                    }
                }
            });
        }

        // Update sidebar badges
        const orderBadge = document.getElementById('sidebar-orders-badge');
        const inquiryBadge = document.getElementById('sidebar-inquiry-badge');
        const transaksiBadge = document.getElementById('sidebar-transaksi-badge');
        const productionsBadge = document.getElementById('sidebar-productions-badge');
        
        // Badge Pesanan
        if (orderBadge) {
            const orderCount = unreadCounts.orders || 0;
            orderBadge.textContent = orderCount > 99 ? '99+' : orderCount.toString();
            if (orderCount > 0) {
                orderBadge.classList.remove('hidden');
                orderBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
            } else {
                orderBadge.classList.add('hidden');
                orderBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
            }
        }
        
        // Badge Penawaran
        if (inquiryBadge) {
            const inquiryCount = unreadCounts.inquiries || 0;
            inquiryBadge.textContent = inquiryCount > 99 ? '99+' : inquiryCount.toString();
            if (inquiryCount > 0) {
                inquiryBadge.classList.remove('hidden');
                inquiryBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
            } else {
                inquiryBadge.classList.add('hidden');
                inquiryBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
            }
        }
        
        // Badge Transaksi
        if (transaksiBadge) {
            const transaksiCount = (unreadCounts.orders || 0) + (unreadCounts.inquiries || 0);
            transaksiBadge.textContent = transaksiCount > 99 ? '99+' : transaksiCount.toString();
            if (transaksiCount > 0) {
                transaksiBadge.classList.remove('hidden');
                transaksiBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
            } else {
                transaksiBadge.classList.add('hidden');
                transaksiBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
            }
        }
        
        // Badge Produksi
        if (productionsBadge) {
            const productionsCount = unreadCounts.productions || 0;
            productionsBadge.textContent = productionsCount > 99 ? '99+' : productionsCount.toString();
            if (productionsCount > 0) {
                productionsBadge.classList.remove('hidden');
                productionsBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
            } else {
                productionsBadge.classList.add('hidden');
                productionsBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
            }
        }
        
        // Badge Kontak
        const contactsBadge = document.getElementById('sidebar-contacts-badge');
        if (contactsBadge) {
            const contactsCount = unreadCounts.contacts || 0;
            console.log('[NotificationSystem] Updating contacts badge with count:', contactsCount);
            contactsBadge.textContent = contactsCount > 99 ? '99+' : contactsCount.toString();
            if (contactsCount > 0) {
                contactsBadge.classList.remove('hidden');
                contactsBadge.removeAttribute('hidden');
                contactsBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
                console.log('[NotificationSystem] Contacts badge SHOWN:', contactsCount, 'Element:', contactsBadge);
            } else {
                contactsBadge.classList.add('hidden');
                contactsBadge.setAttribute('hidden', 'true');
                contactsBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
                console.log('[NotificationSystem] Contacts badge HIDDEN (count is 0)');
            }
        } else {
            console.warn('[NotificationSystem] Contacts badge element not found (ID: sidebar-contacts-badge)');
        }
        
        // Badge Quotations (untuk supervisor)
        const quotationsBadge = document.getElementById('sidebar-quotations-badge');
        if (quotationsBadge) {
            const quotationsCount = unreadCounts.quotations || 0;
            console.log('[NotificationSystem] Updating quotations badge with count:', quotationsCount);
            quotationsBadge.textContent = quotationsCount > 99 ? '99+' : quotationsCount.toString();
            if (quotationsCount > 0) {
                quotationsBadge.classList.remove('hidden');
                quotationsBadge.removeAttribute('hidden');
                quotationsBadge.style.cssText = 'display: inline-flex !important; visibility: visible !important; opacity: 1 !important;';
                console.log('[NotificationSystem] Quotations badge SHOWN:', quotationsCount, 'Element:', quotationsBadge);
            } else {
                quotationsBadge.classList.add('hidden');
                quotationsBadge.setAttribute('hidden', 'true');
                quotationsBadge.style.cssText = 'display: none !important; visibility: hidden !important;';
                console.log('[NotificationSystem] Quotations badge HIDDEN (count is 0)');
            }
        } else {
            console.warn('[NotificationSystem] Quotations badge element not found (ID: sidebar-quotations-badge). Available badges:', {
                productions: !!document.getElementById('sidebar-productions-badge'),
                quotations: !!document.getElementById('sidebar-quotations-badge')
            });
        }
        
        // Update page title
        if (unreadCounts.total > 0) {
            const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
            const pageTitle = isSupervisorPage ? 'Supervisor Panel - POLJAM TECH' : 'Admin Panel - POLJAM TECH';
            document.title = `(${unreadCounts.total}) ${pageTitle}`;
        } else {
            const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
            const pageTitle = isSupervisorPage ? 'Supervisor Panel - POLJAM TECH' : 'Admin Panel - POLJAM TECH';
            document.title = pageTitle;
        }
    }

    createNotificationModal() {
        console.log('[NotificationSystem] Creating notification modal...');
        
        // Hapus modal lama jika ada
        const existingModal = document.getElementById('notification-modal');
        if (existingModal) {
            existingModal.remove();
        }

        const modal = document.createElement('div');
        modal.id = 'notification-modal';
        
        // Responsive positioning untuk mobile dan desktop
        const isMobile = window.innerWidth < 1024;
        const isAdminPage = window.location.pathname.startsWith('/admin') || window.location.pathname.startsWith('/supervisor');
        
        if (isMobile) {
            // Mobile: full width dengan padding kecil
            modal.className = 'fixed inset-x-4 top-20 z-[60] w-auto max-w-sm mx-auto';
        } else {
            // Desktop: fixed right
            modal.className = 'fixed right-4 z-[60] w-full max-w-sm';
        }
        
        // Position di bawah navbar - sesuaikan berdasarkan halaman
        // Di halaman admin/supervisor, navbar lebih pendek (py-4 = 64px), jadi gunakan 72px
        // Di homepage, navbar lebih tinggi (header + navbar = ~119px), jadi gunakan 119px
        // Di mobile, tambahkan sedikit lebih banyak space
        if (isMobile) {
            modal.style.top = isAdminPage ? '80px' : '130px';
        } else {
            modal.style.top = isAdminPage ? '72px' : '119px';
        }
        
        modal.style.display = 'none';
        modal.style.visibility = 'hidden';
        modal.style.opacity = '0';
        modal.style.transition = 'opacity 0.3s ease';
        
        modal.innerHTML = `
            <div class="bg-white dark:bg-gray-800 rounded-xl shadow-2xl border border-gray-200 dark:border-gray-700 max-w-sm w-full max-h-[80vh] flex flex-col overflow-hidden">
                <div class="p-4 border-b border-gray-200 dark:border-gray-700 flex-shrink-0 bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-gray-800 dark:to-gray-800">
                    <div class="flex items-center justify-between">
                        <div class="flex items-center gap-2">
                            <div class="p-1.5 bg-blue-100 dark:bg-blue-900/40 rounded-lg">
                                <i class="fas fa-bell text-blue-600 dark:text-blue-400 text-sm"></i>
                            </div>
                            <h3 class="text-base font-bold text-gray-900 dark:text-white">Notifikasi</h3>
                        </div>
                    <div class="flex items-center space-x-2">
                        <button id="mark-all-read-btn" 
                                class="text-xs bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white px-2.5 py-1 rounded-lg transition-colors cursor-pointer font-medium">
                                Tandai Semua
                        </button>
                        <button onclick="notificationSystem.toggleModal()" 
                                    class="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors p-1.5 rounded-lg hover:bg-gray-100 dark:hover:bg-gray-700"
                            title="Tutup">
                                    <i class="fas fa-times text-base"></i>
                        </button>
                    </div>
                </div>
                </div>
                <div id="notification-content" class="flex-1 overflow-y-auto p-4 dark:bg-gray-800" style="max-height: calc(80vh - 120px);">
                    <div class="text-center py-8">
                        <div class="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 dark:border-blue-400 mx-auto"></div>
                        <p class="text-gray-500 dark:text-gray-400 mt-2 text-sm">Memuat notifikasi...</p>
                    </div>
                </div>
            </div>
        `;

        document.body.appendChild(modal);
        console.log('[NotificationSystem] Modal created');
        
        // Bind event untuk tombol "Tandai Semua Dibaca"
        // Gunakan setTimeout untuk memastikan DOM sudah siap
        setTimeout(() => {
            const markAllReadBtn = document.getElementById('mark-all-read-btn');
            if (markAllReadBtn) {
                // Remove existing listeners by cloning
                const newBtn = markAllReadBtn.cloneNode(true);
                markAllReadBtn.parentNode.replaceChild(newBtn, markAllReadBtn);
                
                newBtn.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    console.log('[NotificationSystem] Mark all read button clicked');
                    console.log('[NotificationSystem] window.notificationSystem:', !!window.notificationSystem);
                    console.log('[NotificationSystem] this:', this);
                    
                    // Gunakan window.notificationSystem jika ada, jika tidak gunakan this
                    const notificationSystem = window.notificationSystem || this;
                    console.log('[NotificationSystem] Using notificationSystem:', !!notificationSystem);
                    console.log('[NotificationSystem] markAllAsRead method exists:', typeof notificationSystem?.markAllAsRead === 'function');
                    
                    if (notificationSystem && typeof notificationSystem.markAllAsRead === 'function') {
                        console.log('[NotificationSystem] Calling markAllAsRead()...');
                        notificationSystem.markAllAsRead();
                    } else {
                        console.error('[NotificationSystem] markAllAsRead method not found', {
                            notificationSystem: !!notificationSystem,
                            hasMarkAllAsRead: typeof notificationSystem?.markAllAsRead,
                            windowNotificationSystem: !!window.notificationSystem
                        });
                        
                        // Fallback: coba panggil langsung dari window
                        if (window.notificationSystem && typeof window.notificationSystem.markAllAsRead === 'function') {
                            console.log('[NotificationSystem] Fallback: calling from window.notificationSystem');
                            window.notificationSystem.markAllAsRead();
                        }
                    }
                });
                console.log('[NotificationSystem] Mark all read button event bound successfully');
            } else {
                console.warn('[NotificationSystem] Mark all read button not found');
            }
        }, 100);
        
        // Load notifications
        this.checkNotifications();
    }

    toggleModal() {
        if (this.isToggling) {
            console.log('[NotificationSystem] Toggle in progress, ignoring...');
            return;
        }
        
        this.isToggling = true;
        console.log('[NotificationSystem] Toggle called, current state:', this.isBadgeOpen);
        
        // Jika di mobile, tutup mobile menu dulu
        const isMobile = window.innerWidth < 1024;
        if (isMobile) {
            // Tutup mobile menu jika terbuka
            const mobileMenu = document.querySelector('.lg\\:hidden.fixed.inset-0');
            if (mobileMenu && mobileMenu.__x && mobileMenu.__x.$data.open) {
                mobileMenu.__x.$data.open = false;
                document.body.style.overflow = '';
                console.log('[NotificationSystem] Mobile menu closed');
            }
        }
        
        let modal = document.getElementById('notification-modal');
        
        // Create modal jika belum ada
        if (!modal) {
            console.log('[NotificationSystem] Creating new modal...');
            this.createNotificationModal();
            modal = document.getElementById('notification-modal');
        }
        
        if (!modal) {
            console.error('[NotificationSystem] Modal element not found!');
            this.isToggling = false;
            return;
        }
        
        if (this.isBadgeOpen) {
            // Tutup modal
            console.log('[NotificationSystem] Closing modal...');
            this.isBadgeOpen = false;
            modal.style.opacity = '0';
            setTimeout(() => {
                if (modal) {
                    modal.style.display = 'none';
                    modal.style.visibility = 'hidden';
                }
                this.isToggling = false;
            }, 300);
        } else {
            // Buka modal
            console.log('[NotificationSystem] Opening modal...');
            this.isBadgeOpen = true;
            modal.style.display = 'block';
            modal.style.visibility = 'visible';
            modal.style.opacity = '0';
            
            // Force reflow
            void modal.offsetHeight;
            
            // Fade in
            requestAnimationFrame(() => {
                if (modal) {
                    modal.style.opacity = '1';
                }
                this.isToggling = false;
            });
            
            // Load notifications setelah modal terbuka
            setTimeout(() => {
                this.checkNotifications();
            }, 100);
        }
    }

    /**
     * Clear localStorage untuk menghapus filter timestamp yang mungkin menyebabkan masalah
     */
    clearLocalStorageFilters() {
        localStorage.removeItem('admin_notifications_marked_read_at');
        localStorage.removeItem('supervisor_notifications_marked_read_at');
        localStorage.removeItem('teknisi_notifications_marked_read_at');
        localStorage.removeItem('user_notifications_marked_read_at');
        console.log('[NotificationSystem] Cleared all notification filters from localStorage');
    }

    updateNotificationContent(data) {
        const content = document.getElementById('notification-content');
        if (!content) {
            console.error('[NotificationSystem] Content element not found');
            return;
        }

        try {
            const { unread_counts, recent_items } = data;
            
            console.log('[NotificationSystem] Updating content:', {
                total: unread_counts?.total || 0,
                recent_notifications: recent_items?.notifications?.length || 0,
                notifications_data: recent_items?.notifications
            });
            
            // Pastikan unread_counts ada
            const totalUnread = unread_counts?.total || 0;
            
            if (totalUnread === 0) {
                content.innerHTML = `
                    <div class="text-center text-gray-500 dark:text-gray-400 py-12">
                        <div class="p-4 bg-green-100 dark:bg-green-900/30 rounded-full w-20 h-20 mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-check-circle text-green-500 dark:text-green-400 text-3xl"></i>
                        </div>
                        <p class="font-semibold text-sm mb-1">Tidak ada notifikasi baru</p>
                        <p class="text-xs text-gray-400 dark:text-gray-500">Semua notifikasi sudah dibaca</p>
                    </div>
                `;
                return;
            }
            
            // Gunakan struktur notifikasi baru dari tabel notifications
            // Pastikan kita hanya menampilkan notifikasi yang benar-benar unread
            const notifications = (recent_items?.notifications || []).filter(notif => !notif.is_read);
            
            console.log('[NotificationSystem] Filtered notifications (unread only):', {
                original_count: recent_items?.notifications?.length || 0,
                unread_count: notifications.length,
                notifications: notifications
            });
            
            if (notifications.length === 0) {
                content.innerHTML = `
                    <div class="text-center text-gray-500 dark:text-gray-400 py-12">
                        <div class="p-4 bg-green-100 dark:bg-green-900/30 rounded-full w-20 h-20 mx-auto mb-4 flex items-center justify-center">
                            <i class="fas fa-check-circle text-green-500 dark:text-green-400 text-3xl"></i>
                        </div>
                        <p class="font-semibold text-sm mb-1">Tidak ada notifikasi baru</p>
                        <p class="text-xs text-gray-400 dark:text-gray-500">Semua notifikasi sudah dibaca</p>
                    </div>
                `;
                return;
            }

            // Group notifications by type
            const groupedNotifications = this.groupNotificationsByType(notifications);
            
            let html = '<div class="space-y-4">';
            
            // Tampilkan notifikasi berdasarkan prioritas: order > inquiry > production > contact
            const priorityOrder = ['order', 'inquiry', 'production', 'contact'];
            
            priorityOrder.forEach(type => {
                if (groupedNotifications[type] && groupedNotifications[type].length > 0) {
                    html += this.renderNotificationGroup(type, groupedNotifications[type]);
                }
            });
            
            html += '</div>';
            content.innerHTML = html;
            
            // Bind click events untuk mark as read
            this.bindNotificationClickEvents();
            
            // Add footer
            this.addNotificationFooter(totalUnread);
            
            console.log('[NotificationSystem] Content updated successfully');
        } catch (error) {
            console.error('[NotificationSystem] Error updating content:', error);
            content.innerHTML = `
                <div class="text-center text-gray-500 dark:text-gray-400 py-4">
                    <i class="fas fa-exclamation-triangle text-yellow-500 text-2xl mb-2"></i>
                    <p>Terjadi kesalahan saat memuat notifikasi</p>
                    <p class="text-xs mt-2">Silakan refresh halaman</p>
                                        </div>
                    `;
                }
            }

    /**
     * Group notifications by type
     */
    groupNotificationsByType(notifications) {
        const grouped = {
            order: [],
            inquiry: [],
            production: [],
            contact: []
        };
        
        notifications.forEach(notif => {
            if (grouped[notif.type]) {
                grouped[notif.type].push(notif);
            }
        });
        
        return grouped;
    }

    /**
     * Render notification group
     */
    renderNotificationGroup(type, notifications) {
        const typeConfig = {
            order: {
                title: 'Pesanan',
                icon: 'fa-shopping-cart',
                color: 'orange',
                route: (notif) => `/admin/master/orders/${notif.related_order_id || notif.data?.order_id || ''}`
            },
            inquiry: {
                title: 'Penawaran',
                icon: 'fa-question-circle',
                color: 'purple',
                route: (notif) => {
                    // Deteksi role dari meta tag atau pathname
                    const userTypeMeta = document.querySelector('meta[name="user-type"]');
                    const userType = userTypeMeta ? userTypeMeta.getAttribute('content') : null;
                    const isSupervisor = userType === 'supervisor' || window.location.pathname.startsWith('/supervisor');
                    
                    if (isSupervisor) {
                        // Untuk supervisor, arahkan ke halaman quotations dengan highlight inquiry ID
                        const inquiryId = notif.related_inquiry_id || notif.data?.inquiry_id || '';
                        return inquiryId ? `/supervisor/quotations?highlight=${inquiryId}` : '/supervisor/quotations';
                    }
                    // Default untuk admin
                    return `/admin/product-inquiries?search=${notif.related_inquiry_id || notif.data?.inquiry_id || ''}`;
                }
            },
            production: {
                title: 'Produksi',
                icon: 'fa-industry',
                color: 'indigo',
                route: (notif) => {
                    // Deteksi role dari meta tag atau pathname
                    const userTypeMeta = document.querySelector('meta[name="user-type"]');
                    const userType = userTypeMeta ? userTypeMeta.getAttribute('content') : null;
                    const isSupervisor = userType === 'supervisor' || window.location.pathname.startsWith('/supervisor');
                    
                    if (isSupervisor) {
                        return `/supervisor/productions/${notif.related_production_id || notif.data?.production_id || ''}`;
                    }
                    // Default untuk admin
                    return `/admin/productions/${notif.related_production_id || notif.data?.production_id || ''}`;
                }
            },
            contact: {
                title: 'Kontak',
                icon: 'fa-envelope',
                color: 'blue',
                route: (notif) => {
                    const contactId = notif.related_contact_id || notif.data?.contact_id || '';
                    // Arahkan ke halaman kontak dengan filter unread atau highlight contact tertentu
                    if (contactId) {
                        return `/admin/contacts?status=unread&highlight=${contactId}`;
                    }
                    return '/admin/contacts?status=unread';
                }
            }
        };
        
        const config = typeConfig[type];
        if (!config) return '';
        
        let html = `<div class="mb-3">
            <div class="flex items-center gap-2 mb-2.5">
                <div class="p-1.5 ${this.getTypeIconBg(config.color)} rounded-lg">
                    <i class="fas ${config.icon} ${this.getTypeIconColor(config.color)} text-sm"></i>
                </div>
                <h4 class="text-sm font-semibold text-gray-800 dark:text-gray-200">${config.title}</h4>
                <span class="text-xs text-gray-500 dark:text-gray-400">(${notifications.length})</span>
            </div>
        </div>`;
        
        notifications.forEach((notif, index) => {
            const url = config.route(notif);
            console.log('[NotificationSystem] Generated URL for notification:', {
                type: type,
                notificationId: notif.id,
                url: url,
                userType: document.querySelector('meta[name="user-type"]')?.getAttribute('content'),
                pathname: window.location.pathname
            });
            const colors = this.getNotificationColors(config.color, notif.subtype);
            const isUnread = !notif.is_read;
                        
                        html += `
                <a href="${url}" 
                   class="block mb-2.5 p-3 ${colors.bg} rounded-lg border ${isUnread ? colors.border + ' border-l-4' : 'border-gray-200 dark:border-gray-600'} ${colors.hover} transition-all duration-200 cursor-pointer notification-item group"
                   data-notification-id="${notif.id}"
                               style="text-decoration: none; color: inherit;">
                    <div class="flex items-start justify-between gap-3">
                        <div class="flex-1 min-w-0">
                            <div class="flex items-start gap-2 mb-1.5">
                                <div class="p-1 ${this.getTypeIconBg(config.color)} rounded mt-0.5 flex-shrink-0">
                                    <i class="fas ${config.icon} ${this.getTypeIconColor(config.color)} text-xs"></i>
                                    </div>
                                <div class="flex-1 min-w-0">
                                    <div class="flex items-center gap-2 flex-wrap mb-1">
                                        <p class="font-semibold text-sm ${colors.text} leading-tight">${notif.title}</p>
                                        ${isUnread ? '<span class="w-2 h-2 bg-blue-500 rounded-full flex-shrink-0"></span>' : ''}
                                        ${this.getSubtypeBadge(notif.subtype, config.color)}
                                    </div>
                                    <p class="text-xs ${colors.text.replace('900', '700').replace('100', '300')} leading-relaxed mb-1.5">
                                        ${notif.message}
                                    </p>
                                    ${this.getNotificationDetails(notif, colors.text)}
                                    <div class="flex items-center gap-2 mt-2 pt-2 border-t ${colors.border.replace('500', '200').replace('border-l-4', '')}">
                                        <span class="text-[10px] ${colors.text.replace('900', '500').replace('100', '400')}">
                                            <i class="fas fa-clock mr-1"></i>${this.formatTime(notif.created_at)}
                                        </span>
                                    </div>
                                    </div>
                                </div>
                                </div>
                        <div class="flex-shrink-0 ${colors.text.replace('900', '400').replace('100', '500')} group-hover:opacity-100 opacity-60 transition-opacity mt-0.5">
                            <i class="fas fa-chevron-right text-xs"></i>
                            </div>
                        </div>
                        </a>
                    `;
        });
        
        return html;
    }

    /**
     * Get notification colors based on type and subtype
     */
    getNotificationColors(baseColor, subtype) {
        const colorMap = {
            orange: {
                bg: 'bg-orange-50 dark:bg-orange-900/20',
                border: 'border-orange-500',
                hover: 'hover:bg-orange-100 dark:hover:bg-orange-900/30 hover:shadow-sm',
                text: 'text-orange-900 dark:text-orange-100'
            },
            purple: {
                bg: 'bg-purple-50 dark:bg-purple-900/20',
                border: 'border-purple-500',
                hover: 'hover:bg-purple-100 dark:hover:bg-purple-900/30 hover:shadow-sm',
                text: 'text-purple-900 dark:text-purple-100'
            },
            indigo: {
                bg: 'bg-indigo-50 dark:bg-indigo-900/20',
                border: 'border-indigo-500',
                hover: 'hover:bg-indigo-100 dark:hover:bg-indigo-900/30 hover:shadow-sm',
                text: 'text-indigo-900 dark:text-indigo-100'
            },
            blue: {
                bg: 'bg-blue-50 dark:bg-blue-900/20',
                border: 'border-blue-500',
                hover: 'hover:bg-blue-100 dark:hover:bg-blue-900/30 hover:shadow-sm',
                text: 'text-blue-900 dark:text-blue-100'
            },
            green: {
                bg: 'bg-green-50 dark:bg-green-900/20',
                border: 'border-green-500',
                hover: 'hover:bg-green-100 dark:hover:bg-green-900/30 hover:shadow-sm',
                text: 'text-green-900 dark:text-green-100'
            },
            cyan: {
                bg: 'bg-cyan-50 dark:bg-cyan-900/20',
                border: 'border-cyan-500',
                hover: 'hover:bg-cyan-100 dark:hover:bg-cyan-900/30 hover:shadow-sm',
                text: 'text-cyan-900 dark:text-cyan-100'
            },
            emerald: {
                bg: 'bg-emerald-50 dark:bg-emerald-900/20',
                border: 'border-emerald-500',
                hover: 'hover:bg-emerald-100 dark:hover:bg-emerald-900/30 hover:shadow-sm',
                text: 'text-emerald-900 dark:text-emerald-100'
            }
        };
        
        // Override colors based on subtype
        if (subtype === 'payment_received') {
            return colorMap.green;
        } else if (subtype === 'final_offer_sent') {
            return colorMap.emerald;
        } else if (subtype === 'inquiry_forwarded_teknisi' || subtype === 'inquiry_forwarded_supervisor') {
            return colorMap.cyan;
        }
        
        return colorMap[baseColor] || colorMap.blue;
    }

    /**
     * Get type icon background color
     */
    getTypeIconBg(color) {
        const bgMap = {
            orange: 'bg-orange-100 dark:bg-orange-900/40',
            purple: 'bg-purple-100 dark:bg-purple-900/40',
            indigo: 'bg-indigo-100 dark:bg-indigo-900/40',
            blue: 'bg-blue-100 dark:bg-blue-900/40',
            green: 'bg-green-100 dark:bg-green-900/40',
            cyan: 'bg-cyan-100 dark:bg-cyan-900/40',
            emerald: 'bg-emerald-100 dark:bg-emerald-900/40'
        };
        return bgMap[color] || bgMap.blue;
    }

    /**
     * Get type icon text color
     */
    getTypeIconColor(color) {
        const textMap = {
            orange: 'text-orange-600 dark:text-orange-400',
            purple: 'text-purple-600 dark:text-purple-400',
            indigo: 'text-indigo-600 dark:text-indigo-400',
            blue: 'text-blue-600 dark:text-blue-400',
            green: 'text-green-600 dark:text-green-400',
            cyan: 'text-cyan-600 dark:text-cyan-400',
            emerald: 'text-emerald-600 dark:text-emerald-400'
        };
        return textMap[color] || textMap.blue;
    }

    /**
     * Get subtype badge
     */
    getSubtypeBadge(subtype, baseColor) {
        const badgeMap = {
            'new_order': '<span class="px-2 py-0.5 bg-orange-100 dark:bg-orange-900/40 text-orange-700 dark:text-orange-300 text-[10px] rounded-md font-medium">Baru</span>',
            'order_created': '<span class="px-2 py-0.5 bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300 text-[10px] rounded-md font-medium">Dibuat</span>',
            'payment_received': '<span class="px-2 py-0.5 bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300 text-[10px] rounded-md font-medium">Pembayaran</span>',
            'new_inquiry': '<span class="px-2 py-0.5 bg-purple-100 dark:bg-purple-900/40 text-purple-700 dark:text-purple-300 text-[10px] rounded-md font-medium">Baru</span>',
            'final_offer_sent': '<span class="px-2 py-0.5 bg-emerald-100 dark:bg-emerald-900/40 text-emerald-700 dark:text-emerald-300 text-[10px] rounded-md font-medium">Final Offer</span>',
            'inquiry_forwarded_teknisi': '<span class="px-2 py-0.5 bg-cyan-100 dark:bg-cyan-900/40 text-cyan-700 dark:text-cyan-300 text-[10px] rounded-md font-medium">Ke Teknisi</span>',
            'inquiry_forwarded_supervisor': '<span class="px-2 py-0.5 bg-indigo-100 dark:bg-indigo-900/40 text-indigo-700 dark:text-indigo-300 text-[10px] rounded-md font-medium">Ke Supervisor</span>',
            'new_contact': '<span class="px-2 py-0.5 bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300 text-[10px] rounded-md font-medium">Baru</span>',
            'production_assigned': '<span class="px-2 py-0.5 bg-indigo-100 dark:bg-indigo-900/40 text-indigo-700 dark:text-indigo-300 text-[10px] rounded-md font-medium">Ditugaskan</span>',
            'planning_submitted': '<span class="px-2 py-0.5 bg-blue-100 dark:bg-blue-900/40 text-blue-700 dark:text-blue-300 text-[10px] rounded-md font-medium">Planning</span>',
            'completion_submitted': '<span class="px-2 py-0.5 bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300 text-[10px] rounded-md font-medium">Completion</span>',
            'completed_and_sold': '<span class="px-2 py-0.5 bg-emerald-100 dark:bg-emerald-900/40 text-emerald-700 dark:text-emerald-300 text-[10px] rounded-md font-medium">Selesai & Sale</span>',
            'production_revised': '<span class="px-2 py-0.5 bg-orange-100 dark:bg-orange-900/40 text-orange-700 dark:text-orange-300 text-[10px] rounded-md font-medium">Direvisi</span>'
        };
        
        return badgeMap[subtype] || '';
    }

    /**
     * Get notification details based on type
     */
    getNotificationDetails(notif, textColor) {
        const data = notif.data || {};
        let details = '';
        const detailColor = textColor.replace('900', '600').replace('100', '400');
        
        if (notif.type === 'order') {
            const detailItems = [];
            if (data.customer_name) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-user text-[10px]"></i><span class="text-[10px]">${data.customer_name}</span></span>`);
            }
            if (data.product_title) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-box text-[10px]"></i><span class="text-[10px]">${data.product_title}</span></span>`);
            }
            if (data.total_price) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-money-bill-wave text-[10px]"></i><span class="text-[10px] font-medium">Rp ${this.formatNumber(data.total_price)}</span></span>`);
            }
            if (data.payment_method) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-${this.getPaymentIcon(data.payment_method)} text-[10px]"></i><span class="text-[10px]">${data.payment_method}</span></span>`);
            }
            if (detailItems.length > 0) {
                details = `<div class="flex flex-wrap items-center gap-2 mt-1.5 ${detailColor}">${detailItems.join('')}</div>`;
            }
        } else if (notif.type === 'inquiry') {
            const detailItems = [];
            if (data.name) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-user text-[10px]"></i><span class="text-[10px]">${data.name}</span></span>`);
            }
            if (data.product_title) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-box text-[10px]"></i><span class="text-[10px]">${data.product_title}</span></span>`);
            }
            if (data.final_price) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-tag text-[10px]"></i><span class="text-[10px] font-medium">Rp ${this.formatNumber(data.final_price)}</span></span>`);
            }
            if (detailItems.length > 0) {
                details = `<div class="flex flex-wrap items-center gap-2 mt-1.5 ${detailColor}">${detailItems.join('')}</div>`;
            }
        } else if (notif.type === 'production') {
            const detailItems = [];
            if (data.customer_name) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-user text-[10px]"></i><span class="text-[10px]">${data.customer_name}</span></span>`);
            }
            if (data.product_title) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-box text-[10px]"></i><span class="text-[10px]">${data.product_title}</span></span>`);
            }
            if (data.order_id) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-shopping-cart text-[10px]"></i><span class="text-[10px] font-medium">Order #${data.order_id}</span></span>`);
            }
            if (detailItems.length > 0) {
                details = `<div class="flex flex-wrap items-center gap-2 mt-1.5 ${detailColor}">${detailItems.join('')}</div>`;
            }
        } else if (notif.type === 'contact') {
            const detailItems = [];
            if (data.nama) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-user text-[10px]"></i><span class="text-[10px]">${data.nama}</span></span>`);
            }
            if (data.email) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-envelope text-[10px]"></i><span class="text-[10px]">${data.email}</span></span>`);
            }
            if (data.subject) {
                detailItems.push(`<span class="inline-flex items-center gap-1"><i class="fas fa-tag text-[10px]"></i><span class="text-[10px]">${data.subject}</span></span>`);
            }
            if (detailItems.length > 0) {
                details = `<div class="flex flex-wrap items-center gap-2 mt-1.5 ${detailColor}">${detailItems.join('')}</div>`;
            }
        }
        
        return details;
    }

    /**
     * Format number with thousand separator
     */
    formatNumber(num) {
        return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ".");
    }

    /**
     * Bind click events untuk mark as read
     */
    bindNotificationClickEvents() {
        const notificationItems = document.querySelectorAll('.notification-item');
        notificationItems.forEach(item => {
            item.addEventListener('click', async (e) => {
                const notificationId = item.dataset.notificationId;
                if (notificationId) {
                    // Mark as read secara async (tidak blocking navigation)
                    this.markNotificationAsRead(notificationId).then(() => {
                        // Refresh badge setelah mark as read
                        setTimeout(() => {
                            this.checkNotifications();
                        }, 300);
                    }).catch(err => {
                        console.error('[NotificationSystem] Error marking notification as read:', err);
                    });
                }
            });
        });
    }

    /**
     * Mark single notification as read
     */
    async markNotificationAsRead(notificationId) {
        try {
            const isAdminPage = window.location.pathname.startsWith('/admin');
            const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
            const isTeknisiPage = window.location.pathname.startsWith('/teknisi');
            const userTypeMeta = document.querySelector('meta[name="user-type"]');
            const userType = userTypeMeta ? userTypeMeta.getAttribute('content') : null;
            
            let markReadUrl;
            if (isAdminPage) {
                markReadUrl = `/admin/notifications/${notificationId}/mark-read`;
            } else if (isSupervisorPage || userType === 'supervisor') {
                markReadUrl = `/supervisor/notifications/${notificationId}/mark-read`;
            } else if (isTeknisiPage || userType === 'teknisi') {
                markReadUrl = `/teknisi/notifications/${notificationId}/mark-read`;
                } else {
                markReadUrl = `/notifications/${notificationId}/mark-read`;
            }
            
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) return;
            
            const response = await fetch(markReadUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                }
            });
            
            const data = await response.json();
            if (data.success && data.data?.unread_counts) {
                // Update badge langsung dengan data dari response
                this.updatePageBadges(data.data.unread_counts);
            }
            
            return data;
        } catch (error) {
            console.error('[NotificationSystem] Error marking notification as read:', error);
            throw error;
        }
    }

    /**
     * Add notification footer
     */
    addNotificationFooter(total) {
            const modal = document.getElementById('notification-modal');
        if (!modal) return;
        
                // Hapus footer lama jika ada
                const oldFooter = modal.querySelector('.notification-footer');
                if (oldFooter) {
                    oldFooter.remove();
                }
                
                // Tambahkan footer baru
                const footer = document.createElement('div');
        footer.className = 'notification-footer p-3 border-t border-gray-200 dark:border-gray-700 flex-shrink-0 bg-gray-50 dark:bg-gray-900/50';
                footer.innerHTML = `
            <div class="flex justify-between items-center text-xs text-gray-500 dark:text-gray-400">
                <span class="font-medium">Total: <span class="text-blue-600 dark:text-blue-400 font-bold">${total}</span> notifikasi baru</span>
                <span class="flex items-center gap-1">
                    <i class="fas fa-sync-alt text-[10px]"></i>
                    <span>${this.formatTime(new Date())}</span>
                </span>
                    </div>
                `;
                
        const content = document.getElementById('notification-content');
                if (content && content.parentNode) {
                    content.parentNode.insertBefore(footer, content.nextSibling);
        }
    }

    async markAllAsRead() {
        try {
            // Deteksi role berdasarkan pathname atau meta tag
            const isAdminPage = window.location.pathname.startsWith('/admin');
            const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
            const isTeknisiPage = window.location.pathname.startsWith('/teknisi');
            const userTypeMeta = document.querySelector('meta[name="user-type"]');
            const userType = userTypeMeta ? userTypeMeta.getAttribute('content') : null;
            
            let markReadUrl;
            if (isAdminPage) {
                markReadUrl = '/admin/notifications/mark-all-read';
            } else if (isSupervisorPage || userType === 'supervisor') {
                markReadUrl = '/supervisor/notifications/mark-all-read';
            } else if (isTeknisiPage || userType === 'teknisi') {
                markReadUrl = '/teknisi/notifications/mark-all-read';
            } else {
                markReadUrl = '/notifications/mark-all-read';
            }
            
            console.log('[NotificationSystem] Marking all as read from:', markReadUrl);
            
            const csrfToken = document.querySelector('meta[name="csrf-token"]');
            if (!csrfToken) {
                console.error('[NotificationSystem] CSRF token not found');
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Error!',
                        text: 'CSRF token tidak ditemukan',
                        icon: 'error',
                        confirmButtonColor: '#ef4444',
                        confirmButtonText: 'OK'
                    });
                }
                return;
            }
            
            const response = await fetch(markReadUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken.getAttribute('content'),
                    'Accept': 'application/json'
                },
                body: JSON.stringify({ type: 'all' })
            });

            if (!response.ok) {
                const errorText = await response.text();
                console.error('[NotificationSystem] Response not OK:', response.status, errorText);
                throw new Error(`Network response was not ok: ${response.status}`);
            }

            const data = await response.json();
            console.log('[NotificationSystem] Mark all read response:', data);
            
            if (data.success) {
                // Hapus localStorage timestamp karena kita tidak lagi menggunakan filter timestamp
                // Notifikasi akan di-filter berdasarkan is_read di database saja
                const isAdminPage = window.location.pathname.startsWith('/admin');
                const isSupervisorPage = window.location.pathname.startsWith('/supervisor');
                const isTeknisiPage = window.location.pathname.startsWith('/teknisi');
                const userTypeMeta = document.querySelector('meta[name="user-type"]');
                const userType = userTypeMeta ? userTypeMeta.getAttribute('content') : null;
                
                // Hapus timestamp dari localStorage untuk menghindari konflik
                if (isAdminPage || userType === 'admin') {
                    localStorage.removeItem('admin_notifications_marked_read_at');
                    console.log('[NotificationSystem] Removed admin marked_read_at timestamp from localStorage');
                } else if (isSupervisorPage || userType === 'supervisor') {
                    localStorage.removeItem('supervisor_notifications_marked_read_at');
                    console.log('[NotificationSystem] Removed supervisor marked_read_at timestamp from localStorage');
                } else if (isTeknisiPage || userType === 'teknisi') {
                    localStorage.removeItem('teknisi_notifications_marked_read_at');
                    console.log('[NotificationSystem] Removed teknisi marked_read_at timestamp from localStorage');
                } else {
                    localStorage.removeItem('user_notifications_marked_read_at');
                    console.log('[NotificationSystem] Removed user marked_read_at timestamp from localStorage');
                }
                
                // Update badges langsung dengan data dari response
                // Response sudah mengembalikan unread_counts yang sudah di-update
                const updatedCounts = data.data?.unread_counts || { 
                    total: 0,
                    orders: 0,
                        inquiries: 0,
                    contacts: 0,
                    productions: 0
                    };
                    this.updatePageBadges(updatedCounts);
                
                // Update modal content jika masih terbuka
                if (this.isBadgeOpen) {
                    // Set unread_counts ke 0 karena semua sudah di-mark as read
                    this.updateNotificationContent({
                        unread_counts: updatedCounts,
                        recent_items: {
                            notifications: []
                        }
                    });
                }
                
                // Tampilkan notifikasi sukses
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Berhasil!',
                        text: `Semua notifikasi telah ditandai sebagai dibaca (${data.data?.updated_count || 0} notifikasi)`,
                        icon: 'success',
                        iconColor: '#10b981',
                        confirmButtonColor: '#3b82f6',
                        confirmButtonText: 'OK',
                        timer: 2000,
                        timerProgressBar: true,
                        allowOutsideClick: false,
                        allowEscapeKey: false,
                        showClass: {
                            popup: 'animate__animated animate__fadeInDown animate__faster'
                        },
                        hideClass: {
                            popup: 'animate__animated animate__fadeOutUp animate__faster'
                        }
                    });
                }
                
                // Deteksi jika user sedang di halaman orders atau inquiries
                const currentPath = window.location.pathname;
                const isOrdersPage = currentPath.includes('/admin/master/orders') || currentPath.includes('/admin/transaksi/sales');
                const isInquiriesPage = currentPath.includes('/admin/product-inquiries');
                
                // Refresh notifications setelah 500ms untuk memastikan database sudah ter-update
                // Dan untuk mendapatkan data terbaru dari server
                setTimeout(() => {
                    console.log('[NotificationSystem] Refreshing notifications after mark all read...');
                    this.checkNotifications();
                    
                    // Jika di halaman orders atau inquiries, refresh halaman untuk update status di tabel
                    if (isOrdersPage || isInquiriesPage) {
                        console.log('[NotificationSystem] Refreshing page to update table indicators (is_read status)');
                        // Refresh setelah SweetAlert timer selesai (2 detik) + sedikit delay
                        setTimeout(() => {
                            window.location.reload();
                        }, 2500);
                    } else {
                        // Double check setelah 1 detik untuk memastikan badge ter-update
                        setTimeout(() => {
                            this.checkNotifications();
                        }, 1000);
                    }
                }, 500);
            } else {
                console.error('[NotificationSystem] Mark all read failed:', data.message);
                if (typeof Swal !== 'undefined') {
                    Swal.fire({
                        title: 'Gagal!',
                        text: data.message || 'Gagal menandai semua notifikasi sebagai dibaca',
                        icon: 'error',
                        confirmButtonColor: '#ef4444',
                        confirmButtonText: 'OK'
                    });
                }
            }
        } catch (error) {
            console.error('[NotificationSystem] Error marking notifications as read:', error);
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    title: 'Error!',
                    text: 'Terjadi kesalahan saat menandai notifikasi: ' + error.message,
                    icon: 'error',
                    confirmButtonColor: '#ef4444',
                    confirmButtonText: 'OK'
                });
            }
        }
    }


    formatTime(dateString) {
        if (!dateString) return 'N/A';
        
        const date = new Date(dateString);
        const now = new Date();
        const diffInMinutes = Math.floor((now - date) / (1000 * 60));
        
        if (diffInMinutes < 1) return 'Baru saja';
        if (diffInMinutes < 60) return `${diffInMinutes} menit lalu`;
        
        const diffInHours = Math.floor(diffInMinutes / 60);
        if (diffInHours < 24) return `${diffInHours} jam lalu`;
        
        const diffInDays = Math.floor(diffInHours / 24);
        return `${diffInDays} hari lalu`;
    }

    getPaymentIcon(paymentMethod) {
        const icons = {
            'transfer': 'university',
            'midtrans': 'credit-card',
            'cod': 'money-bill-wave',
            'dp': 'hand-holding-usd',
            'lunas': 'check-circle',
            'tunai': 'money-bill',
        };
        return icons[paymentMethod] || 'credit-card';
    }

    bindEvents() {
        // Bind event ke semua tombol notifikasi (desktop & mobile) dengan retry mechanism yang lebih robust
        const tryBind = () => {
            // Cari semua button dengan id 'notification-toggle-btn' (bisa ada 2: desktop & mobile)
            const notificationBtns = document.querySelectorAll('#notification-toggle-btn');
            
            if (notificationBtns.length > 0) {
                let boundCount = 0;
                
                notificationBtns.forEach((notificationBtn, index) => {
                    // Check if already bound (avoid duplicate listeners)
                    if (notificationBtn.dataset.bound === 'true') {
                        console.log(`[NotificationSystem] Button ${index + 1} already bound, skipping...`);
                        boundCount++;
                        return;
                    }
                    
                    // Remove existing listeners by cloning
                    const newBtn = notificationBtn.cloneNode(true);
                    notificationBtn.parentNode.replaceChild(newBtn, notificationBtn);
                    
                    // Mark as bound
                    newBtn.dataset.bound = 'true';
                    
                    // Tambahkan class untuk identifikasi
                    newBtn.classList.add('notification-toggle-btn-bound');
                    
                    newBtn.addEventListener('click', (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        console.log(`[NotificationSystem] Notification button ${index + 1} clicked (mobile: ${window.innerWidth < 1024})`);
                        
                        // Jika di mobile menu, tutup mobile menu dulu
                        if (window.innerWidth < 1024) {
                            const mobileMenu = document.querySelector('[x-data*="open"]');
                            if (mobileMenu && mobileMenu.__x) {
                                mobileMenu.__x.$data.open = false;
                                document.body.style.overflow = '';
                            }
                        }
                        
                        if (window.notificationSystem) {
                            window.notificationSystem.toggleModal();
                        } else {
                            console.error('[NotificationSystem] notificationSystem instance not found!');
                        }
                    }, { passive: false });
                    
                    boundCount++;
                    console.log(`[NotificationSystem] Event bound to notification button ${index + 1} successfully`);
                });
                
                if (boundCount > 0) {
                    console.log(`[NotificationSystem] Total ${boundCount} notification button(s) bound`);
                    return true;
                }
                return false;
            } else {
                console.warn('[NotificationSystem] Notification button not found, will retry...');
                return false;
            }
        };
        
        // Try immediately
        if (!tryBind()) {
            // Retry after a short delay if button not found
            setTimeout(() => {
                if (!tryBind()) {
                    // Retry with longer delay
                    setTimeout(() => {
                        if (!tryBind()) {
                            // Use MutationObserver to watch for button appearance
                            const observer = new MutationObserver((mutations, obs) => {
                                if (tryBind()) {
                                    obs.disconnect();
                                }
                            });
                            
                            observer.observe(document.body, {
                                childList: true,
                                subtree: true
                            });
                            
                            // Stop observing after 5 seconds
                            setTimeout(() => {
                                observer.disconnect();
                                if (document.querySelectorAll('#notification-toggle-btn').length === 0) {
                                    console.error('[NotificationSystem] Notification button never found after all retries!');
                                }
                            }, 5000);
                        }
                    }, 500);
                }
            }, 100);
        }
    }
}

// Initialize notification system
console.log('[NotificationSystem] Script loaded, document readyState:', document.readyState);
console.log('[NotificationSystem] Current path:', window.location.pathname);
console.log('[NotificationSystem] Is admin page?', window.location.pathname.startsWith('/admin'));

// Skip initialization di halaman profile
const currentPath = window.location.pathname;
if (currentPath === '/profile' || currentPath.startsWith('/profile/')) {
    console.log('[NotificationSystem] Profile page detected, skipping notification system initialization');
} else {
if (document.readyState === 'loading') {
    console.log('[NotificationSystem] DOM still loading, waiting for DOMContentLoaded...');
    document.addEventListener('DOMContentLoaded', function() {
        console.log('[NotificationSystem] DOMContentLoaded fired, initializing...');
        if (!window.notificationSystem) {
            console.log('[NotificationSystem] Creating new NotificationSystem instance...');
            window.notificationSystem = new NotificationSystem();
        } else {
            console.log('[NotificationSystem] NotificationSystem already exists, skipping...');
        }
    });
} else {
    console.log('[NotificationSystem] DOM already ready, initializing immediately...');
    if (!window.notificationSystem) {
        console.log('[NotificationSystem] Creating new NotificationSystem instance...');
        window.notificationSystem = new NotificationSystem();
    } else {
        console.log('[NotificationSystem] NotificationSystem already exists, skipping...');
        }
    }
}
