<?php

namespace Tests\Feature;

use Tests\TestCase;
use App\Models\Product;
use App\Models\Category;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;

class ProductCreateTest extends TestCase
{
    use RefreshDatabase;

    public function test_product_create_form_loads()
    {
        // Create admin user
        $admin = User::factory()->create([
            'user_type' => 'admin'
        ]);

        // Create a category
        $category = Category::create([
            'name' => 'Test Category',
            'type' => 'service',
            'description' => 'Test description'
        ]);

        $response = $this->actingAs($admin)->get('/admin/master/products/create');
        
        $response->assertStatus(200);
        $response->assertViewIs('admin.master.products.create');
    }

    public function test_product_can_be_created()
    {
        // Create admin user
        $admin = User::factory()->create([
            'user_type' => 'admin'
        ]);

        // Create a category
        $category = Category::create([
            'name' => 'Test Category',
            'type' => 'service',
            'description' => 'Test description'
        ]);

        $productData = [
            'product_title' => 'Test Service',
            'product_description' => 'Test description',
            'product_prices' => 150000,
            'category_id' => $category->id,
            'product_work_duration' => '3 hari',
            'product_garansi' => '6 bulan',
            'service_fee' => 50000,
        ];

        $response = $this->actingAs($admin)->post('/admin/master/products', $productData);
        
        $response->assertRedirect('/admin/master/products');
        $response->assertSessionHas('add_message', 'Produk berhasil ditambahkan!');
        
        $this->assertDatabaseHas('products', [
            'product_title' => 'Test Service',
            'product_prices' => 150000,
            'category_id' => $category->id,
        ]);
    }

    public function test_product_creation_requires_authentication()
    {
        $response = $this->get('/admin/master/products/create');
        $response->assertRedirect('/login');
    }

    public function test_product_creation_requires_admin_role()
    {
        $user = User::factory()->create([
            'user_type' => 'user'
        ]);

        $response = $this->actingAs($user)->get('/admin/master/products/create');
        $response->assertStatus(401);
    }
}
